%function [rho,kf,visc] = BatzleWang(T,P,fluid,varargin)
%
% Compute fluid density, bulk modulus and viscosity using Batzle-Wang rock 
% physics equations
%
% Reference: Batzle, M. and Wang, Z., 1992. Seismic properties of pore
% fluids: Geophysics 57, 1396 - 1408
%
% INPUTS:   T = temperature (deg C)
%           P = pressure (MPa)
%           fluid = fluid type, can be 'gas','oil','brine','CO2'
%
% OPTIONAL INPUTS:
%           For Gas:
%               'G' = specific gas gravity (0.5<G<1.5)
%           For Oil:
%               'API' = oil API density (10<API<75)
%               'live' = logical - true for live oil, false for dead
%           For live oil:
%               'G' = specific gas gravity (0.5<G<1.5)
%           For brine:
%               'conc' = brine salinity (ppm)
%           For CO2:
%               No optional arguments used
%
% OUTPUTS:
%           rho = fluid density
%           kf = fluid bulk modulus
%           visc = fluid viscosity
%
% Written by J.P. Verdon, 2014
% University of Bristol
%

function [rho,kf,visc] = BatzleWang(T,P,fluid,varargin)

% Process the optional arguments (overwriting defaults where defined)
iarg = 1;
while iarg <= (length(varargin))
    switch lower(varargin{iarg})
        case 'g'
            G = varargin{iarg+1};
            iarg = iarg + 2;
            
        case 'api'
            API = varargin{iarg+1};
            iarg = iarg + 2;
            
        case 'live'
            live = varargin{iarg+1};
            iarg = iarg + 2;
            
        case 'conc'
            S = varargin{iarg+1};
            iarg = iarg + 2;
            
        otherwise
            error('BW:UnknownOption',...
                ['Unknown option: ',varargin{iarg}]);
    end
end



switch lower(fluid)
    % GAS
    case 'gas'
        if ~exist('G','var')
            G=input('Specific gas gravity (0.5<G<1.5): ');
        end
        
        R=8.31;
        T=T+273;
	
        % Check that T/P values are not too close to a critical point
        Ppr=P/(4.892-0.4048*G);
        Tpr=T/(94.72+170.75*G);
	
        if Ppr < 1.1 && Ppr > 0.9  
            error('BW:CriticalPoint',['SUPRISE, COCKFACE, Ppr is too close to 1 and Doug is GAY',...
                char(10),'Ppr = ',num2str(Ppr)])
        end
        if Tpr < 1.1 && Tpr > 0.9  
            error('BW:CriticalPoint',['SUPRISE, COCKFACE, Tpr is too close to 1 and Doug is GAY',...
                char(10),'Tpr = ',num2str(Tpr)])
        end
	
        E=0.109*(3.85-Tpr)*(3.85-Tpr)*exp(-(0.45+(8*(0.56-1/Tpr)*(0.56-1/Tpr)))*(Ppr^1.2)/Tpr);  
        Z=(0.03+0.00527*(3.5-Tpr)*(3.5-Tpr)*(3.5-Tpr))*Ppr + (0.642*Tpr-0.007*Tpr*Tpr*Tpr*Tpr-0.52) + E; 
        rho=1000*28.8*G*P/(Z*R*T);

        gamma=0.85+(5.6/(Ppr+2))+(27.1/((Ppr+3.5)*(Ppr+3.5)))-8.7*exp(-0.65*(Ppr+1));
        dZ=(0.03+0.00527*(3.5-Tpr)*(3.5-Tpr)*(3.5-Tpr)) + 0.109*1.6*((0.56-1/Tpr)*(0.56-1/Tpr)/Tpr)*(3.85-Tpr)*(3.85-Tpr)*(Ppr^0.2)*exp(-(0.45+(8*(0.56-1/Tpr)*(0.56-1/Tpr)))*(Ppr^1.2)/Tpr);
        kf=P*gamma/(1-Ppr*dZ/Z);
	  
        etaatm=0.0001*(Tpr*(28+48*G-5*G*G)-(6.47/(G*G))+(35/G)+1.14*G-15.55);
        eta=0.001*Ppr*etaatm*(((1057-8.08*Tpr)/Ppr)+((796*sqrt(Ppr)-704)/(((Tpr-1)^0.7)*(Ppr+1)))-3.24*Tpr-38);
	  
        kf=kf*1e6;
        visc=eta*1e-3;

        
        
    % OIL
    case 'oil'
        if ~exist('API','var')
            API=input('Enter API (10<API<75)');
        end
        
        rhos=141.5/(API+131.5);
        rhop=rhos+(0.00277*P-(1.71e-7)*P*P*P)*(rhos-1.15)*(rhos-1.15)+(3.49e-4)*P;
        rho=(1e3)*rhop/(0.972+(3.81e-4)*((T+17.78)^1.175));
        
	  
        vo=15450/(sqrt(77.1+API))-3.7*T+4.64*P+0.0115*(0.36*sqrt(API)-1)*T*P;
        ako=vo*vo*rho;

        gamma=10^(5.693-2.863/rhos);
        etat=-1+10^(0.505*gamma*((17.8+T)^-1.163));
        ai=10^(18.6*(0.1*log10(etat)+((log10(etat)+2)^-0.1)-0.985));
        eta=etat+0.145*P*ai;

        rho=rho;
        visc=eta*1e-3;
        kf=ako;
        
        if ~exist('live','var')
            live = false;
        end
        
        if live
            if ~exist('G','var')
                G=input('Specific gas gravity (0.5<G<1.5): ');
            end
            rg=2.03*G*((P*exp(0.02878*API-0.00377*T))^1.205);
            Bo=0.972+0.00038*((2.4*rg*sqrt(G/rhos)+T+17.8)^1.175);
            rhol=(rhos+0.0012*G*rg)/Bo;
            rholp=rhol+(0.00277*P-(1.71e-7)*P*P*P)*(rhol-1.15)*(rhol-1.15)+(3.49e-4)*P;
            rholo=rholp/(0.972+(3.81e-4)*((T+17.78)^1.175));
            rho=rholo*1e3;
            
            aklo=vo*vo*rholo;
            ako=aklo;
  
            rgsc=2.03*G*((0.1*exp(0.02878*API-0.00377*15.6))^1.205);
            Bosc=0.972+0.00038*((2.4*rgsc*sqrt(G/rhol)+15.6+17.8)^1.175);
            rholsc=(rhos+0.0012*G*rg)/Bo;
            gamma2=10^(5.693-2.863/rholsc);
            etat2=-1+10^(0.505*gamma2*((17.8+T)^-1.163));
            ai2=10^(18.6*(0.1*log10(etat2)+((log10(etat2)+2)^-0.1)-0.985));
            etao2=etat2+0.145*P*ai2;
            
            kf=ako/1e9;
            visc=etao2*1e-3;
        end
        
    % Brine    
    case 'brine'
        if ~exist('S','var')
          	S=input('Enter brine conc (ppm): ');
        end
        
        S=S/1e6;
    
        rhow=1.0+(1e-6)*(-80*T - 3.3*T*T + 0.00175*T*T*T + 489*P - 2*T*P + 0.016*T*T*P - 1.3*(1e-5)*T*T*T*P - 0.333*P*P - 0.002*T*P*P);
        rhob=rhow+S*(0.668+0.44*S+(300*P-2400*P*S+T*(80+3*T-3300*S-13*P+47*P*S))*1e-6);
        rhob=rhob*1e3;
    
        v(1,1)=1402.85;
        v(2,1)=4.871*T;
        v(3,1)=-0.04783*T*T;
        v(4,1)=T*T*T*1.487e-4;
        v(5,1)=T*T*T*T*-2.197e-7;
        v(1,2)=P*1.524;
        v(2,2)=-0.0111*T*P;
        v(3,2)=T*T*P*2.747e-4;
        v(4,2)=T*T*T*P*(-6.502e-7);
        v(5,2)=T*T*T*T*P*7.987e-10;
        v(1,3)=P*P*3.437e-3;
        v(2,3)=T*P*P*1.739e-4;
        v(3,3)=T*T*P*P*(-2.135e-6);
        v(4,3)=T*T*T*P*P*(-1.455e-8);
        v(5,3)=T*T*T*T*P*P*5.230e-11;
        v(1,4)=P*P*P*(-1.197e-5);
        v(2,4)=T*P*P*P*(-1.628e-6);
        v(3,4)=T*T*P*P*P*1.237e-8;
        v(4,4)=T*T*T*P*P*P*1.327e-10;
        v(5,4)=T*T*T*T*P*P*P*(-4.614e-13);
    
        vw=0.0;
        for i=1:5;
            for j=1:4;
                vw=vw+v(i,j);
            end
        end
    
        vb=vw+S*(1170-9.6*T+0.055*T*T-T*T*T*(8.5e-5)+2.6*P-0.0029*T*P-0.0476*P*P) + (780-10*P+0.16*P*P)*(S^1.5) - 820*S*S;
        akb=vb*vb*rhob;
    
        etab=0.1+0.333*S+(1.65+91.9*S*S*S)*exp(-(0.42*((S^0.8)-0.17)*((S^0.8)-0.17)+0.045)*T^0.8);

        rho=rhob;
        kf=akb/1e9;
        visc=etab*1e-3;

    % CO2    
    case 'co2'

        R=8.31;
        T=T+273;
        G=1.5349;
        Ppc=7.4;
        Tpc=31.1+273;
    
        Ppr=P/Ppc;
        Tpr=T/Tpc;
    
        if Ppr < 1.1 && Ppr > 0.9  
            error('BW:CriticalPoint',['SUPRISE, COCKFACE, Ppr is too close to 1 and Doug is GAY',...
                char(10),'Ppr = ',num2str(Ppr)])
        end
        if Tpr < 1.1 && Tpr > 0.9  
            error('BW:CriticalPoint',['SUPRISE, COCKFACE, Tpr is too close to 1 and Doug is GAY',...
                char(10),'Tpr = ',num2str(Tpr)])
        end

        E=0.109*(3.85-Tpr)*(3.85-Tpr)*exp(-(0.45+(8*(0.56-1/Tpr)*(0.56-1/Tpr)))*(Ppr^1.2)/Tpr);
        Z=(0.03+0.00527*(3.5-Tpr)*(3.5-Tpr)*(3.5-Tpr))*Ppr + (0.642*Tpr-0.007*Tpr*Tpr*Tpr*Tpr-0.52) + E;
        rhog=1000*28.8*G*P/(Z*R*T);
    
        gamma=0.85+(5.6/(Ppr+2))+(27.1/((Ppr+3.5)*(Ppr+3.5)))-8.7*exp(-0.65*(Ppr+1));
        dZ=(0.03+0.00527*(3.5-Tpr)*(3.5-Tpr)*(3.5-Tpr)) + 0.109*1.6*((0.56-1/Tpr)*(0.56-1/Tpr)/Tpr)*(3.85-Tpr)*(3.85-Tpr)*(Ppr^0.2)*exp(-(0.45+(8*(0.56-1/Tpr)*(0.56-1/Tpr)))*(Ppr^1.2)/Tpr);
        akg=P*gamma/(1-Ppr*dZ/Z);
    
        etaatm=0.0001*(Tpr*(28+48*G-5*G*G)-(6.47/(G*G))+(35/G)+1.14*G-15.55);
        etag=0.001*Ppr*etaatm*(((1057-8.08*Tpr)/Ppr)+((796*sqrt(Ppr)-704)/(((Tpr-1)^0.7)*(Ppr+1)))-3.24*Tpr-38);
	  
        rho=rhog;
        kf=akg/1e3;
        visc=etag*1e-3;
    
    otherwise
        error('BW:UnknownOption',...
        	['Unknown option: ',fluid]);
end
