%------------------------------------------------------------------------------
%
%     Constrained cbs curve fitting in tension
%     Nick Teanby 30/01/07
%
%------------------------------------------------------------------------------
%
%     Function to smooth a set of unevenly spaced x,y data
%	and output the cubic B spline parameters and covariance.
%
%     Allows constraints to be imposed on the smooth curve
%     by using the method of Lagrange multipliers.
%
%	Constraints [optional] can be on y, dy/dx, or d2y/dx2.
%
%	NB. curvature at ends of curve will be constrained to zero by default.
%
%     Tension is applied using a quadratic spring approximation as explained
%	in Teanby 2007.
%
%	  input
%       -----
%	x	float(n)		x data
%	y	float(n)		y data
%	dy	float(n)		y data errors
%	M	int			number of splines to use
%	gamma	float			tension
%
%	  input [optional]
%	  ----------------
%	x0	float(n0)		x constraints
%	y0	float(n0)		constraints [ y , dy/dx, d2y/dx2 ]
%	ctype	float(n0)		constraint type [0=y, 1=grad, 2=curvature]
%
%       output
%       ------
%	m	float(M)		spline parameters
%	cm	float(M,M)		covariance matrix of spline parameters
%
%------------------------------------------------------------------------------
function [m,Cm] = fcbsfit_tension(x,y,dy,M,gamma,x0,y0,ctype)

% number of data points
N    = length(y);

% knot spacing
k    = (x(N)-x(1))/(M-3);

% spacing and number of points in the quadrature grid
DX   = k/5.;
Q    = ceil( (x(N)-x(1))/DX + 1 );

% set up A matrix
for i=1:N
  for j=1:M
    A(i,j)=fspline(j,x(i),x(1),k);
  end
end

% set up Ce matrix
Ce = zeros(N,N);
for i=1:N
  Ce(i,i)=dy(i)^2;
end

% set up F matrix and h vector for constraints
if nargin == 5
  % no user defined constraints specified
  NC = 2;
  F=zeros(2,M);
  % constrain ends of curve to have zero curvature
  F(1,1)   = fspline_curv(  1,x(1),x(1),k);
  F(1,2)   = fspline_curv(  2,x(1),x(1),k);
  F(1,3)   = fspline_curv(  3,x(1),x(1),k);
  F(2,M-2) = fspline_curv(M-2,x(N),x(1),k);
  F(2,M-1) = fspline_curv(M-1,x(N),x(1),k);
  F(2,M)   = fspline_curv(  M,x(N),x(1),k);
  h(1)     = 0.;
  h(2)     = 0.;
elseif nargin == 8
  N0 = length(x0);
  NC = 2 + N0;
  F=zeros(NC,M);
  % constrain ends of curve to have zero curvature
  F(1,1)   = fspline_curv(  1,x(1),x(1),k);
  F(1,2)   = fspline_curv(  2,x(1),x(1),k);
  F(1,3)   = fspline_curv(  3,x(1),x(1),k);
  F(2,M-2) = fspline_curv(M-2,x(N),x(1),k);
  F(2,M-1) = fspline_curv(M-1,x(N),x(1),k);
  F(2,M)   = fspline_curv(  M,x(N),x(1),k);
  h(1) = 0.;
  h(2) = 0.;
  % user supplied contraints
  for i=1:N0
    if ctype(i)     == 0
      for j=1:M
        F(i+2,j) = fspline(j,x0(i),x(1),k);
      end
      h(i+2) = y0(i);
    elseif ctype(i) == 1
      for j=1:M
        F(i+2,j) = fspline_grad(j,x0(i),x(1),k);
      end
      h(i+2) = y0(i);
    elseif ctype(i) == 2
      for j=1:M
        F(i+2,j) = fspline_curv(j,x0(i),x(1),k);
      end
      h(i+2) = y0(i);
    else
      fprintf('ERROR: unknown constraint type')
    end
  end
else
  fprintf('ERROR: wrong number of arguments')
end

% set up D matrix
for q=1:Q
  xq = x(1) + (q-1)*DX;
  for j=1:M
    D(q,j) = fspline_grad(j,xq,x(1),k);
  end
end

% set up inverse theory matrices
E = A'*inv(Ce)*A + gamma*D'*D;
G1 = [E,F';F,zeros(NC:NC)];
G2 = [A'*inv(Ce)*y;h'];

% invert
mm=inv(G1)*G2;
% spline coeffs are first M params
m=mm(1:M);

% model parameter covariance matrix
Cm = inv(E) - inv(E)*F'*inv(F*inv(E)*F')*F*inv(E);




