%---------------------------------------------------------
% cbsfit_tradeoff
%---------------------------------------------------------
%
% read in some example datasets and plot a tradeoff curve
% of chi2 for various values of the tension (gamma)
% using the method of Teanby 2007
%
% set by user
% -fname	select differnt datasets
% -M		number of spline parameters for the fit
%		large values of M give more detail
% -ngamma	number of tensions to search
% -gammas	array of tensions to search
%
%
%---------------------------------------------------------
% N.Teanby	17/01/07
%---------------------------------------------------------

% clear old variables
close
clear

fprintf('---------------------------------------------\n');
fprintf('- CBSFIT_TRADEOFF: N. Teanby 2007           -\n');
fprintf('- plot a tradeoff curve for                 -\n');
fprintf('- constrained spline smoothing with tension -\n');
fprintf('---------------------------------------------\n');

% get input params from user and load data
dfile   = input('Enter input file (x,y,dy):','s');
data    = dlmread(deblank(dfile(1,:)));
x       = data(:,1);
y       = data(:,2);
dy      = data(:,3);
% number of splines
M       = input('Enter number of parameters for fit (M):');
M       = round(M);
% number of points in tradeoff curve
ngamma  = input('Enter number of points in tradeoff curve (ngamma):');
ngamma  = round(ngamma);
% tension step
gamma_step = input('Enter tension step (gamma_step):');
% constraints
copt    = input('use a constraints file (y/n):','s');
if copt == 'y'
  cfile   = input('Enter constraints file (x0,y0,ctype):','s');
  cdata   = dlmread(deblank(cfile(1,:)));
  x0      = cdata(:,1);
  y0      = cdata(:,2);
  ctype   = cdata(:,3);
end

% set up array of tensions to plot tradeoff curve for
for l=1:ngamma
  gammas(l) = (l-1.0)*gamma_step;
end

% number of data points
N = length(x);
% knot spacing
k=(x(N)-x(1))/(M-3);

% loop over different tensions and calc chi2 each time
% to build up the trade off curve
for l = 1:ngamma
  % fit
  if copt == 'y'
    % user constraints are defined
    [m,Cm] = fcbsfit_tension(x,y,dy,M,gammas(l),x0,y0,ctype);
  else
    % no user constraints are defined
    [m,Cm] = fcbsfit_tension(x,y,dy,M,gammas(l));
  end
  % find fitted curve at x positions
  % set up A matrix
  for i=1:N
    for j=1:M
      A(i,j) = fspline(j,x(i),x(1),k);
    end
  end
  % fitted curve at x positions and covaraince of fitted curve
  f = A*m;
  Cf = A*Cm*A';
  % set errors to diagonal elements of covariance matrix
  for i=1:N
    if Cf(i,i) < 0
      fprintf('error: negative diag cov\n')
    end
    df(i) = sqrt(Cf(i,i));
  end
  % calc misfit for each gamma
  chi2(l) = 0.;
  for i=1:N
    chi2(l) = chi2(l) + (f(i)-y(i))^2 /(dy(i)^2);
  end
end

% decide on chi2_target
% based on how well the zero tension case fits the data
if chi2(1)/N > 1.0
  chi2_target = chi2(1) + chi2(1)/N;
  % lots of outliers
  % and chi2/n is never under 1, so aim for chi2=chi2(1)+chi2(1)/N
else
  chi2_target = N;
  % well behaved cases, so aim for chi2=N
end


% plot tradeoff curve
plot(gammas,chi2/N,'-x')
hold on
plot([0,gammas(l)],[chi2_target/N,chi2_target/N],'-r')
hold off
xlabel('gamma')
ylabel('chi2/N')

% print out gamma -v- misfit
[gammas',chi2',chi2'/N]

% find value of gamma that gives chi2 ~ chi2_target
lopt = 1;
diffchi2 = abs(chi2_target - chi2(1));
for l=2:ngamma
  diffchi2_test = abs(chi2_target - chi2(l));
  if diffchi2_test < diffchi2
    diffchi2 = diffchi2_test;
    lopt = l;
  end
end
fprintf('recommend gamma=%16.10f\n',gammas(lopt))

% interpolate trade off curve to get more accurate value of gamma
gamma_int = -1.0;
for l=1:ngamma-1
  if chi2(l) < chi2_target
    if chi2(l+1) > chi2_target 
      grad      = (chi2(l+1) - chi2(l))/(gammas(l+1)-gammas(l));
      gamma_int = gammas(l) + (chi2_target - chi2(l))/ grad;
    end
  end
end
if gamma_int >= 0.0
  fprintf('interpolated gamma=%16.10f\n',gamma_int)
end

% write tradeoff curve gamma -v- chi2/n
fid = fopen(strcat(deblank(dfile(1,:)),'.chi2n'),'w');
for l=1:ngamma
  fprintf(fid,'%12.8f %12.8f\n',gammas(l),chi2(l)/N);
end
fclose(fid);

% write tradeoff curve gamma -v- chi2
fid = fopen(strcat(deblank(dfile(1,:)),'.chi2'),'w');
for l=1:ngamma
  fprintf(fid,'%12.8f %12.8f\n',gammas(l),chi2(l));
end
fclose(fid);

