%---------------------------------------------------------
% cbsfit
%---------------------------------------------------------
%
% read in some example datasets and plot the CBS fit
% using the method of Teanby 2007
%
%
% set by user
% -M		number of spline parameters for the fit
%		large values of M give more detail
% -gamma	the tension to apply to the curve
%		(can be found with cbsfit_tradeoff)
%
%
%---------------------------------------------------------
% N.Teanby	17/01/07	original
% N.Teanby	21/02/07	added option for RMS errorbars
%---------------------------------------------------------

% clear old variables
close
clear

fprintf('---------------------------------------------\n');
fprintf('- CBSFIT: N. Teanby 2007                    -\n');
fprintf('- constrained spline smoothing with tension -\n');
fprintf('---------------------------------------------\n');

% get input params from user and load data
dfile   = input('Enter input file (x,y,dy):','s');
data    = dlmread(deblank(dfile(1,:)));
x       = data(:,1);
y       = data(:,2);
dy      = data(:,3);
% number of splines
M       = input('Enter number of parameters for fit (M):');
M       = round(M);
% tension
gamma   = input('Enter tension for the fit (gamma):');
% errorbars
etype   = input('Enter type of errorbars to use:\n 1) from covariance matrix \n 2) from covariance matrix scaled by chi2 \n 3) rms scatter \n');
% constraints
copt    = input('use a constraints file (y/n):','s');
if copt == 'y'
  cfile   = input('Enter constraints file (x0,y0,ctype):','s');
  cdata   = dlmread(deblank(cfile(1,:)));
  x0      = cdata(:,1);
  y0      = cdata(:,2);
  ctype   = cdata(:,3);
end
% output file
ofile   = input('Enter output file (x,f,df):','s');
ofile   = deblank(ofile(1,:));

% number of points in the output fitted curve (5 gives reasonable sampling)
Q = M*5;

% calculate the spline parameters (m) and the covariance (Cm)
if copt == 'y'
  % user constraints are defined
  [m,Cm] = fcbsfit_tension(x,y,dy,M,gamma,x0,y0,ctype);
else
  % no user constraints are defined
  [m,Cm] = fcbsfit_tension(x,y,dy,M,gamma);
end

% number of data points
N = length(x);

% knot spacing
k=(x(N)-x(1))/(M-3);

% plot the fit f at Q points xq
dx  = (x(N)-x(1))/(Q-1);
f   = zeros(Q);
xq  = zeros(Q,1);

% set up D matrix
for q=1:Q
  xq(q,1) = x(1) + (q-1)*dx;
  for j=1:M
    D(q,j) = fspline(j,xq(q,1),x(1),k);
  end
end

% calc fitted curve using spline coefficients
f  = D*m;
Cf = D*Cm*D';

% calc the error on the fitted curve
for q=1:Q
  if Cf(q,q) < 0
    fprintf('error: negative diag cov\n')
  end
  df(q) = sqrt(Cf(q,q));
end

% calc chi2 (misfit)
% find fitted curve at x positions of the intial data points
% set up A matrix
for i=1:N
  for j=1:M
    A(i,j) = fspline(j,x(i),x(1),k);
  end
end
f1   = A*m;
% calc misfit
chi2 = 0.;
for i=1:N
  chi2 = chi2 + (f1(i)-y(i))^2 /(dy(i)^2);
end
fprintf('chi2   = %16.10f\n',chi2)
fprintf('chi2/n = %16.10f\n',chi2/N)

% calc error on fit 
if etype == 1
  % errorbar is from covaraince matrix
  dfout = df;
elseif etype == 2
  % calc corrected error on fit by scaling by chi2/n for data points
  % within one knot spacing of each x value
  dfout = df;
  for q=1:Q
    ncorr    = 0.0;
    chi2corr = 0.0;
    for i=1:N
      if (x(i)>=xq(q)-k)
        if (x(i)<=xq(q)+k)
          ncorr    = ncorr    + 1;
          chi2corr = chi2corr + (f1(i)-y(i))^2 /(dy(i)^2);
        end
      end
    end
    if (ncorr > 0)
      chi2corr   = chi2corr/ncorr;
    else
      chi2corr   = 1.0;
    end
    if (chi2corr > 1.0)
      dfout(q) = dfout(q)*sqrt(chi2corr);
    end
  end
elseif etype == 3
  % calc error weighted rms scatter of data points from curve
  % within one knot spacing of each x value
  for q=1:Q
    nrms = 0;
    wrms = 0.0;
    rms  = 0.0;
    for i=1:N
      if (x(i)>=xq(q)-k)
        if (x(i)<=xq(q)+k)
          nrms = nrms + 1;
          wrms = wrms + 1.0/(dy(i)^2);
          rms  = rms + (f1(i)-y(i))^2 /(dy(i)^2);
        end
      end
    end
    if (nrms > 0)
      rms   = rms/wrms;
    else
      rms   = 0.0;
    end
    dfout(q) = sqrt(rms);
  end
else
  % errorbar is from covaraince matrix
  disp('ERROR: etype not recognised - defaulting to covariance')
  dfout = df;
end

% write output
fid = fopen(ofile,'w');
for i=1:Q
  fprintf(fid,'%16.6e %16.6e %16.6e\n',xq(i),f(i),dfout(i));
end
fclose(fid);

% plot fit
% fit
plot(xq,f,'-b')
hold on
errorbar(xq,f,dfout,'ob')
% data
errorbar(x,y,dy,'ok')
plot(x,y,'o','MarkerEdgeColor','k','MarkerFaceColor','k')
hold off

