;-----------------------------------------------------------------------
pro ztriangulate,x,y,n,connectivity
;-----------------------------------------------------------------------
;
;
; in:
;  x(n)     fltarr  Longitudes in degrees (MUST BE IN RANGE -180 - 180)
;  y(n)     fltarr  Latitudes
;  n        int     number of lat.long points
; out:
;  connectivity struct  [i].m  = number of points connected to ith vertex
;                       [i].con = intarr of indices of connected points
;
; NOTE: on output the order of points will be rearranged and will be
;  modified slightly by the addition of a small random number to aid
;  the triangulation process
;
;-----------------------------------------------------------------------
;   N. Teanby   19-07-04    Original Code
;-----------------------------------------------------------------------

; set = 1 to plot diagnostic graphs
test_plots = 0

; check ranges
for i=0L,n-1 do begin
  if (abs(x(i)) gt 180.) or (abs(y(i)) gt 90.) then begin
    print,x(i),y(i)
    print,'ERROR: lat/long not in range -90,90/-180,180'
    read,ans
  endif
endfor
; use IDL routine for Delaunay triangulation
; sphere=s returns the connectivity array.
; add a small random number to each vertex prior ro the triangulation
; to aid convergence of the 'triangulate' routine
x1 = x(0:n-1) + randomn(666, n) * 0.01
y1 = y(0:n-1) + randomn(6666, n) * 0.01
; make sure these random numbers dont violate lat=-90,90, long=-180,180
for i=0L,n-1 do begin
  if (x1(i) gt  180.) then x1(i) =  180.
  if (x1(i) lt -180.) then x1(i) = -180.
  if (y1(i) gt   90.) then y1(i) =  90.
  if (y1(i) lt  -90.) then y1(i) = -90.
endfor

; use IDL routine for Delaunay triangulation
; sphere=s returns the connectivity array.
triangulate,x1,y1,tr,sphere=s,/degrees

; the maximum possible number of connected points
mmax = max(s.iend(1:n-1)-s.iend(0:n-2))

; set up the connectivity array
;  -m = number of connected points
;  -con = indices of the connected points
connectivity = replicate({m:0L,con:lonarr(mmax)},n)
ans=''
ntri=n_elements(tr)/3
for i=0L,n-1 do begin
   if (i eq 0) then begin
        ibeg = 0
        iend = s.iend(0) - 1
   endif else begin
        ibeg = s.iend(i-1)
        iend = s.iend(i) - 1
   endelse
   index = s.iadj(ibeg:iend)-1
   connectivity[i].m   = n_elements(index)
   connectivity[i].con = index
   if (test_plots eq 1) then begin
     ;plot the triangles
     map_set,0,0,proj=10,/horizon,/grid
     for j=0L,ntri -1 do begin
       t = [tr[*,j],tr[0,j]]
       plots,x1[t],y1[t]
     endfor
     ; plot vertex numbers
     for j=0L,n-1 do begin
;       print,x1(j),y1(j),string(j)
       xyouts,x1(j),y1(j),strcompress(string(j),/remove_all),alignment=0.5
     endfor
     ; plot centre and connected verticies
     plots,x1(i),y1(i),psym=5
     plots,x1(index),y1(index),psym=2
     ans=''
     read,ans
   endif
endfor

; overwrite x and y as connectivity applies to the rearranged output
; from 'triangulate'
x=x1
y=y1

return
end
