;-----------------------------------------------------------------------
pro ztesselation_varstep,mode,level,step,longrid,latgrid,n
;-----------------------------------------------------------------------
;
; Program to tesselate a sphere with triangles
; - start with an octahedron or icosahedron circumscribed by a unit sphere
; - divide the triangles into (step)^2 equal new triangles
; - project the new vertices onto the unit sphere
; - repeat however many times you want to obtain the required resolution
;
; - the variable step size allows a greater
; range of bin spacings. if step = [2,2,2,2,2] this will
; bisect the triangle each time. steps of 3 and 5 are also valid
;
; NB. 
; 1) work in cartesians, projecting onto a unit sphere, at 
; each iteration to avoid pole 'issues'
; 2) icosahedron gives more even sampling than the octahedron, however
; the octahedron has a smaller difference between levels/ gives a different
; grid which may be useful in order to obtain a specific resolution.
; 3) resulting points are evenly spaced in great circle angle, not lat/lon
; which means that poles are handled correctly.
;
; in:
;    mode    int     8 or 20[default] specify octahedron or icosahedron
;    level   int     number of subdivisions (0 = 0 subdivisions)
;    step    intarr  array of subdivision steps. array length must be
;                    greater than or equal to number of levels-1
;  out:
;    lat/longrid    fltarr(n)   grid of unique lat long points which
;                               are evenly spaced on the sphere
;    n              int         number of lat/long grid points         
;
;
;
;-----------------------------------------------------------------------
;   N. Teanby   13-01-04    Original Code - modified from ztesselation
;-----------------------------------------------------------------------
n=0L
pi=3.141592654
rad2deg=180./pi

; pad step array with 2's so it has at least level elements
while (n_elements(step) lt level) do begin
  step = [step,2]
endwhile

; calc number of triangles
ntriangles=0L
if mode eq 8 then begin
   ntriangles = 8L
endif else begin
   ntriangles = 20L
endelse
if level gt 0 then begin
  ; level 1 is one subdivision
  for i=0,level-1 do begin
    ntriangles = ntriangles  * step(i)^2
  endfor
endif

single_triangle={a:[0.0,0.0,0.0],b:[0.0,0.0,0.0],c:[0.0,0.0,0.0]}

triangle     = replicate(single_triangle,ntriangles)
triangle_new = replicate(single_triangle,ntriangles)
latgrid      = fltarr(ntriangles)
longrid      = fltarr(ntriangles)
latgrid_uniq = fltarr(ntriangles)
longrid_uniq = fltarr(ntriangles)

if (mode eq 8) then begin
   ; start with an octahedron
   ; 6 indices of the starting octahedron (x=1,y=0 is lat=0 lon=0)
   oct1 = [ 1.,  0.,  0.]
   oct2 = [ 0.,  1.,  0.]
   oct3 = [-1.,  0.,  0.]
   oct4 = [ 0., -1.,  0.]
   oct5 = [ 0.,  0.,  1.]
   oct6 = [ 0.,  0., -1.]
   
   ; the 8 vertex coords of the starting triangles
   triangle[0].a = oct1 & triangle[0].b = oct2 & triangle[0].c = oct5
   triangle[1].a = oct2 & triangle[1].b = oct3 & triangle[1].c = oct5
   triangle[2].a = oct3 & triangle[2].b = oct4 & triangle[2].c = oct5
   triangle[3].a = oct5 & triangle[3].b = oct4 & triangle[3].c = oct1
   triangle[4].a = oct2 & triangle[4].b = oct1 & triangle[4].c = oct6
   triangle[5].a = oct3 & triangle[5].b = oct2 & triangle[5].c = oct6
   triangle[6].a = oct4 & triangle[6].b = oct3 & triangle[6].c = oct6
   triangle[7].a = oct6 & triangle[7].b = oct1 & triangle[7].c = oct4
   n=8L
endif else begin
   ; start with an icosahedron
   ; golden ratio
   phi = 2.*cos(pi/5.)
   ; 12 indices of the starting icosahedron
   ico0  = [ 0. , phi ,  1.]
   ico1  = [ 0. ,-phi ,  1.]
   ico2  = [ 0. , phi , -1.]
   ico3  = [ 0. ,-phi , -1.]
   ico4  = [ 1. , 0. , phi ]
   ico5  = [-1. , 0. , phi ]
   ico6  = [ 1. , 0. ,-phi ]
   ico7  = [-1. , 0. ,-phi ]
   ico8  = [ phi, 1. , 0. ]
   ico9  = [-phi, 1. , 0. ]
   ico10 = [ phi,-1. , 0. ]
   ico11 = [-phi,-1. , 0. ]
   ; the 20 vertex coords of the starting triangles
   triangle[0].a  = ico1  & triangle[0].b  = ico3  & triangle[0].c  = ico10
   triangle[1].a  = ico4  & triangle[1].b  = ico1  & triangle[1].c  = ico10
   triangle[2].a  = ico8  & triangle[2].b  = ico4  & triangle[2].c  = ico10
   triangle[3].a  = ico6  & triangle[3].b  = ico8  & triangle[3].c  = ico10
   triangle[4].a  = ico10 & triangle[4].b  = ico6  & triangle[4].c  = ico3
   triangle[5].a  = ico3  & triangle[5].b  = ico1  & triangle[5].c  = ico11
   triangle[6].a  = ico5  & triangle[6].b  = ico11 & triangle[6].c  = ico1
   triangle[7].a  = ico1  & triangle[7].b  = ico4  & triangle[7].c  = ico5
   triangle[8].a  = ico0  &  triangle[8].b  = ico5 & triangle[8].c  = ico4
   triangle[9].a  = ico4  &  triangle[9].b  = ico8 & triangle[9].c  = ico0
   triangle[10].a = ico2  & triangle[10].b = ico0  & triangle[10].c = ico8
   triangle[11].a = ico8  & triangle[11].b = ico6  & triangle[11].c = ico2
   triangle[12].a = ico7  & triangle[12].b = ico2  & triangle[12].c = ico6
   triangle[13].a = ico6  & triangle[13].b = ico3  & triangle[13].c = ico7
   triangle[14].a = ico11 & triangle[14].b = ico7  & triangle[14].c = ico3
   triangle[15].a = ico11 & triangle[15].b = ico5  & triangle[15].c = ico9
   triangle[16].a = ico5  & triangle[16].b = ico0  & triangle[16].c = ico9
   triangle[17].a = ico0  & triangle[17].b = ico2  & triangle[17].c = ico9
   triangle[18].a = ico2  & triangle[18].b = ico7  & triangle[18].c = ico9
   triangle[19].a = ico9  & triangle[19].b = ico11 & triangle[19].c = ico7
   ; normalise the vertices to be on the unit sphere
   for j=0,19 do begin
     triangle[j].a = triangle[j].a/norm(triangle[j].a)
     triangle[j].b = triangle[j].b/norm(triangle[j].b)
     triangle[j].c = triangle[j].c/norm(triangle[j].c)
   endfor
   n=20L
endelse

; split each triangle into 4 new triangles
if level gt 0 then begin
for lev=0,level-1 do begin
  k=0L
; for each triangle
  for i=0L,n-1 do begin
   case 1 of
     step(lev) eq 2: begin
       ; divide into 4 triangles (ie 2^2)
       ; the vertices of the original triangle
       a = triangle[i].a
       b = triangle[i].b
       c = triangle[i].c
       ; the 3 midpoints of each side
       p = (a + b)/2.
       q = (b + c)/2.
       r = (c + a)/2.
       ; normalise onto surface of unit sphere
       p = p/norm(p)
       q = q/norm(q)
       r = r/norm(r)
       ; the 4 new triangles which replace the old triangle
       triangle_new[k].a   = a & triangle_new[k].b   = p & triangle_new[k].c   = r 
       triangle_new[k+1].a = p & triangle_new[k+1].b = b & triangle_new[k+1].c = q
       triangle_new[k+2].a = r & triangle_new[k+2].b = q & triangle_new[k+2].c = c 
       triangle_new[k+3].a = q & triangle_new[k+3].b = r & triangle_new[k+3].c = p
       k = k + 4
     end
    step(lev) eq 3: begin
       ; divide into 9 triangles (ie 3^2)
       ; the vertices of the original triangle
       a = triangle[i].a
       b = triangle[i].b
       c = triangle[i].c
       ; the vertices and midpoints of the subdivided triangle
       p1 = (2*a + b)/3.
       p2 = (a + 2*b)/3.
       q1 = (2*b + c)/3.
       q2 = (b + 2*c)/3.
       r1 = (2*c + a)/3.
       r2 = (c + 2*a)/3.
       m1 = (a + b + c)/3.
       ; normalise onto surface of unit sphere
       p1 = p1/norm(p1)
       p2 = p2/norm(p2)
       q1 = q1/norm(q1)
       q2 = q2/norm(q2)
       r1 = r1/norm(r1)
       r2 = r2/norm(r2)
       m1 = m1/norm(m1)
       ; the 9 new triangles which replace the old triangle
       triangle_new[k].a   = a  & triangle_new[k].b   = p1 & triangle_new[k].c   = r2 
       triangle_new[k+1].a = p1 & triangle_new[k+1].b = p2 & triangle_new[k+1].c = m1
       triangle_new[k+2].a = p2 & triangle_new[k+2].b = b  & triangle_new[k+2].c = q1 
       triangle_new[k+3].a = q1 & triangle_new[k+3].b = q2 & triangle_new[k+3].c = m1
       triangle_new[k+4].a = c  & triangle_new[k+4].b = r1 & triangle_new[k+4].c = q2
       triangle_new[k+5].a = r1 & triangle_new[k+5].b = r2 & triangle_new[k+5].c = m1 
       triangle_new[k+6].a = r2 & triangle_new[k+6].b = p1 & triangle_new[k+6].c = m1
       triangle_new[k+7].a = m1 & triangle_new[k+7].b = p2 & triangle_new[k+7].c = q1
       triangle_new[k+8].a = q2 & triangle_new[k+8].b = r1 & triangle_new[k+8].c = m1
       k = k + 9
     end
    step(lev) eq 5: begin
       ; divide into 9 triangles (ie 3^2)
       ; the vertices of the original triangle
       a = triangle[i].a
       b = triangle[i].b
       c = triangle[i].c
       ; the vertices and midpoints of the subdivided triangle
       p1 = (4*a + b)/5.
       p2 = (3*a + 2*b)/5.
       p3 = (2*a + 3*b)/5.
       p4 = (a + 4*b)/5.
       q1 = (4*b + c)/5.
       q2 = (3*b + 2*c)/5.
       q3 = (2*b + 3*c)/5.
       q4 = (b + 4*c)/5.
       r1 = (4*c + a)/5.
       r2 = (3*c + 2*a)/5.
       r3 = (2*c + 3*a)/5.
       r4 = (c + 4*a)/5.
       m1 = (3*r4 + q1)/4.
       m2 = (2*r4 + 2*q1)/4.
       m3 = (r4   + 3*q1)/4.
       m4 = (2*r3 + q2)/3.
       m5 = (r3 + 2*q2)/3.
       m6 = (r2 + q3)/2.
       ; normalise onto surface of unit sphere
       p1 = p1/norm(p1)
       p2 = p2/norm(p2)
       p3 = p3/norm(p3)
       p4 = p4/norm(p4)
       q1 = q1/norm(q1)
       q2 = q2/norm(q2)
       q3 = q3/norm(q3)
       q4 = q4/norm(q4)
       r1 = r1/norm(r1)
       r2 = r2/norm(r2)
       r3 = r3/norm(r3)
       r4 = r4/norm(r4)
       m1 = m1/norm(m1)
       m2 = m2/norm(m2)
       m3 = m3/norm(m3)
       m4 = m4/norm(m4)
       m5 = m5/norm(m5)
       m6 = m6/norm(m6)
       ; the 25 new triangles which replace the old triangle
       triangle_new[k].a    = a  & triangle_new[k].b    = p1 & triangle_new[k].c    = r4
       triangle_new[k+1].a  = p1 & triangle_new[k+1].b  = p2 & triangle_new[k+1].c  = m1
       triangle_new[k+2].a  = p2 & triangle_new[k+2].b  = p3 & triangle_new[k+2].c  = m2
       triangle_new[k+3].a  = p3 & triangle_new[k+3].b  = p4 & triangle_new[k+3].c  = m3
       triangle_new[k+4].a  = p4 & triangle_new[k+4].b  = b  & triangle_new[k+4].c  = q1
       triangle_new[k+5].a  = q1 & triangle_new[k+5].b  = q2 & triangle_new[k+5].c  = m3
       triangle_new[k+6].a  = q2 & triangle_new[k+6].b  = q3 & triangle_new[k+6].c  = m4
       triangle_new[k+7].a  = q3 & triangle_new[k+7].b  = q4 & triangle_new[k+7].c  = m5
       triangle_new[k+8].a  = q4 & triangle_new[k+8].b  = c  & triangle_new[k+8].c  = r1
       triangle_new[k+9].a  = r1 & triangle_new[k+9].b  = r2 & triangle_new[k+9].c  = m5
       triangle_new[k+10].a = r2 & triangle_new[k+10].b = r3 & triangle_new[k+10].c = m6
       triangle_new[k+11].a = r3 & triangle_new[k+11].b = r4 & triangle_new[k+11].c = m1
       triangle_new[k+12].a = r4 & triangle_new[k+12].b = a  & triangle_new[k+12].c = p1
       triangle_new[k+13].a = m1 & triangle_new[k+13].b = p2 & triangle_new[k+13].c = m2
       triangle_new[k+14].a = m2 & triangle_new[k+14].b = p3 & triangle_new[k+14].c = m3
       triangle_new[k+15].a = m3 & triangle_new[k+15].b = p4 & triangle_new[k+15].c = q1
       triangle_new[k+16].a = m4 & triangle_new[k+16].b = m3 & triangle_new[k+16].c = q2
       triangle_new[k+17].a = m5 & triangle_new[k+17].b = m4 & triangle_new[k+17].c = q3
       triangle_new[k+18].a = r1 & triangle_new[k+18].b = m5 & triangle_new[k+18].c = q4
       triangle_new[k+19].a = r2 & triangle_new[k+19].b = m6 & triangle_new[k+19].c = m5
       triangle_new[k+20].a = r3 & triangle_new[k+20].b = m1 & triangle_new[k+20].c = m6
       triangle_new[k+21].a = m6 & triangle_new[k+21].b = m1 & triangle_new[k+21].c = m2
       triangle_new[k+22].a = m4 & triangle_new[k+22].b = m2 & triangle_new[k+22].c = m3
       triangle_new[k+23].a = m5 & triangle_new[k+23].b = m6 & triangle_new[k+23].c = m4
       triangle_new[k+24].a = m6 & triangle_new[k+24].b = m2 & triangle_new[k+24].c = m4
       k = k + 25
     end
     else: begin
       print,'ERROR: ztesselation_varstep: invalid step value!'
       ans=' '
       read,ans
     end
   endcase
  endfor
  triangle = triangle_new
  n = k
endfor
endif

; calc latitude/longitude grid using the first vertex of each triangle
for k=0L,n-1 do begin
   x = triangle[k].a(0)
   y = triangle[k].a(1)
   z = triangle[k].a(2)
   latgrid[k] = rad2deg * atan(z,sqrt(x^2 + y^2))
   longrid[k] = rad2deg * atan(y,x)
endfor
; now, all the vertices in the grid will not be unique and will be shared by a number of triangles. we only require the uniq values
; need to remove none unique points from latgrid/longrid

; sort by latitude first (this speeds up the next bit loads)
isort = sort(latgrid)
lat_sort = latgrid(isort)
lon_sort = longrid(isort)

TINY = 1.e-3
count =1L
latgrid_uniq[count-1] = lat_sort(0)
longrid_uniq[count-1] = lon_sort(0)
for k=1L,n-1 do begin
   skip = 0
   for j=0L,count-1 do begin
      if abs(lat_sort(k)-latgrid_uniq(j)) le TINY then begin
         if abs(lon_sort(k)-longrid_uniq(j)) le TINY then begin
            skip = 1
         endif
      endif
   endfor
   if (skip eq 0 ) then begin
      count = count + 1
      latgrid_uniq[count-1] = lat_sort(k)
      longrid_uniq[count-1] = lon_sort(k)
   endif
endfor

; redefine lat and long grids
n=count
latgrid = fltarr(n)
longrid = fltarr(n)
latgrid = latgrid_uniq(0:n-1)
longrid = longrid_uniq(0:n-1)  
 
;map_set,0,0,proj=10,/horizon,/grid
;plots,longrid,latgrid,psym=1

return
end
