;-----------------------------------------------------------------------
function finsides,plon,plat,px,py,pz,$
polylon,polylat,polyx,polyy,polyz,n,$
midpolylon,midpolylat,midpolyx,midpolyy,midpolyz
;-----------------------------------------------------------------------
; calculate if point P(plon,plat) lies inside the polygon defined by points
; V(polylon(n),polylat(n)) with midpoint M(midpolylon,midpolylat)
; on the surface of a sphere.
; All points are assumed to be in degrees
;
; method:
;  - calc the winding number 'w', the number of times that the polygon 
;   encircles the point.
;  - w = 1/2pi sum (theta_i) i=0....n-1
;         theta_i = the angle between (polylon_i,polylat_i) and
;                (polylon_i+1,polylat_i+1)
;  - if w eq 1 then px,py enclose plon,plat
;  - use dot product to find theta_i
;  - use cross product to find sign of theta_i
;
;  extra bits for surface of a sphere (to handle the poles + cyclicity)
;  - project all points onto the plane tangent to midpoint of polygon otherwise
;    the angles won't add up to 360 because of curvature of the sphere
;  - if any of vertices are separated by more than 90deg from the midpoint
;    then the polygon is illegal
;  - if angle between P and M is gt max angle between M and Vi  then assume
;    point is outside the polygon. this saves a lot of calculations.
;
;  in:
;   plon,plat       flt         [P] lat/long coords of test point
;   px,py,pz        flt         [P] cartesian coords of test point
;   polylon/lat(n)  fltarr(n)   [V] lat/long coords of polygon in degrees
;   polylx/y/z      fltarr(n)   [V] cartesian coords of polygon in degrees
;   midpolylon/lat  flt         [M] lat/long coords of polygon midpoint
;   midpolyx/y/z    flt         [M] cartesian coords of polygon midpoint
;   n               int         number of points in polygon
;  return value:
;   inside          int      0  outside
;                            1  inside
;                           -1  an error occurred
;
;
;-----------------------------------------------------------------------
;  Nick Teanby  19-7-04 Original Code
;  Nick Teanby  16-8-05 Made code more efficient and use haversine formula
;                       to calculate the distance between midpoint and 
;                       vertices.
;  Nick Teanby  6-1-06  Made code more efficient by precomputing xyz coords
;                       prior to binning and using dot product instead of
;                       haversine formaula
;-----------------------------------------------------------------------
pi=3.141592654
rad2deg=180./pi
deg2rad=pi/180.
inside = -1

; calc great circle angular separations of P and vertices from the midpoint
; using the dot product
psep = acos( px*midpolyx + py*midpolyy + pz*midpolyz )*rad2deg
vsep = fltarr(n)
for i=0,n-1 do begin
  vsep(i) = acos( polyx(i)*midpolyx + polyy(i)*midpolyy + polyz(i)*midpolyz )*rad2deg
endfor
vsep_max = max(vsep)



; check the polygon is not illegal
; (a polygon is legal if the angular separation between each vertex and
; the midpoint is less than 90 degrees)
if (vsep_max lt 90.) then begin
   ; test for simple case first to speed up.
   ; check if separation between P and M is > vsep_max for each vertex
   ; if it is then the point must be outside the polygon as the great
   ; circle lines between vertices project onto the tangent plane 
   ; as straight lines. (this step is to increase efficiency)
   if (psep gt vsep_max) then begin
      inside = 0
      return, inside
   endif
   ; cartesian coords on the surface of unit sphere
   ; the point
   p = fltarr(3)
   p(0) = px
   p(1) = py
   p(2) = pz
   ; the polygon vertices
   v=fltarr(3,n)
   for i=0,n-1 do begin
      v(0,i) = polyx(i)
      v(1,i) = polyy(i)
      v(2,i) = polyz(i)
   endfor
   ; the polygon midpoint
   m=fltarr(3)
   m(0) = midpolyx
   m(1) = midpolyy
   m(2) = midpolyz
   ; to project onto tangent plane to polygon midpoint point, scale each point
   ; by the dot product of M and the V or P (scale polygon vertices and P)
   ; SCALE P
   dot = m ## transpose(p)
   p1=fltarr(3)
   p1(0) = p(0)/dot
   p1(1) = p(1)/dot
   p1(2) = p(2)/dot
   ; SCALE Polygon
   v1=fltarr(3,n+1)
   for i=0,n-1 do begin
      dot = m ## transpose(v(*,i))
      v1(0,i) = v(0,i)/dot
      v1(1,i) = v(1,i)/dot
      v1(2,i) = v(2,i)/dot
   endfor
   ;set last point = first point to close the polygon
   v1(*,n) = v1(*,0)
   ; calc winding number
   theta = fltarr(n)
   vec0=fltarr(3)
   vec1=fltarr(3)
   for i=0,n-1 do begin
     vec0 = v1(*,i)   - p1
     vec1 = v1(*,i+1) - p1
;    check if polygon vertex and point are coincident
;    if they are then count the point as inside
     if (norm(vec0)*norm(vec1) eq 0.0) then begin
       inside = 1
       return,inside
     endif
     dot   = (vec0 ## transpose(vec1))/ (norm(vec0)*norm(vec1))
;    allow for numerical inaccuracies making abs(dot) gt 1
     if dot ge 0. then begin
       dot = min([dot,1.0])
     endif else begin
       dot = max([dot,-1.0])
     endelse
     cross = crossp(vec0,vec1)
;    use dot1 to find the sense of the angle theta
     dot1 = cross ##  transpose(p1)
     if dot1 ge 0 then begin
        theta(i) =  abs(acos(dot))
     endif else begin
        theta(i) = -abs(acos(dot))
     endelse
   endfor
   w = total(theta)
   ; if winding number is ge 2pi then the point is inside
   ; (use a low accuracy value for pi to account for numerical precision)
   if (abs(w) ge 2*3.141) then begin
     inside = 1
     return,inside
   endif else begin
     inside = 0
     return,inside
   endelse
endif else begin
   print,'WARNING: illegal polygon'
   inside = -1
   return,inside
endelse
   
end































