% function [b,a,berr,Mmin,pcdf,mcdf] = BValue(M,...)
%
% Compute G-R b-value for a population of earthquake magnitudes. Contains a
% range of possible methods. Slope can be fitted explicitly in log-N M
% space, or using the Aki max likelihood method. 
%
% The minimum magnitude of completeness can be computed using
% Kolmogorov-Smirnov, Wiemer/Wyss, or can be defined explicitly
%
% INPUTS:   M = event magnitudes
%
% OPTIONAL INPUTS:
%           mminmethod = method used to compute mmin. Can be 'ks' for
%               Kolmogorov-Smirnov, 'ww' for Weimer/Wyss, or set to a
%               number for use of an explicit value
%  
%           rounded = true/false, data are rounded into (usually M0.1) bins
%
%           fit = method used to fit data, can be 'ml' for max likelihood
%               or 'log' to fit data in log-N M space
%
%           figures = true/false, plot results
%
%           E = cut-off for WW/KS Mmin fitting
%
%           error = values at which error bars are computed
%
%           bins = bin values at which to put data (otherwise will use
%               sort(M) - irregular bins with 1 event per bin
%
%           Nmin = minimum number of events to return a valid result
%
% OUTPUTS:  b = b value
%           a = a value
%           berr = error in b value
%           Mmin = minimum magnitude of completeness
%           pcdf = observed freq-mag distribution
%           mcdf = modelled freq-mag distribution
%
% Written by J.P. Verdon, UoBristol, 2014
%
%

function [b,a,berr,Mmin,pcdf,mcdf,ecdf] = BValue(M,varargin)

% Deal with the occasional empty M vector
if isempty(M)
    fprintf('%s \n','Error: M is empty')
    b=NaN; a=NaN; berr=[NaN NaN]; Mmin=NaN; pcdf=NaN; mcdf=NaN; ecdf=NaN;
    return
end


% If ML is a row vector, turn it into a column
if size(M,2) > size(M,1)
    M=M';
end


% Process defaults and optional arguments
mminmethod = 'ks';
rounded = false;
fit = 'ml';
fig = false;
E = 0.1;
err = NaN;
bins = sort(M);
ibin = false;
Nmin = 35;
ierror = 0;
ecdf = NaN;

iarg = 1;
while iarg <= (length(varargin))
    switch lower(varargin{iarg})
        
        case 'mminmethod'
            mminmethod = varargin{iarg+1};
            iarg = iarg + 2;
        
        case 'rounded'
            rounded = varargin{iarg+1};
            iarg = iarg + 2;
        
        case 'fit'
            fit = varargin{iarg+1};
            iarg = iarg + 2;
        
        case 'e'
            E = varargin{iarg+1};
            iarg = iarg + 2;
        
        case 'error'
            err = varargin{iarg+1};
            iarg = iarg + 2;
        
        case 'figure'
            fig = varargin{iarg+1};
            iarg = iarg + 2;
        
        case 'bins'
            bins = varargin{iarg+1};
            iarg = iarg + 2;
            ibin = true;
            
        case 'nmin'
            Nmin = varargin{iarg+1};
            iarg = iarg + 2;
            
        otherwise
            error('BValue:UnknownOption',...
                ['Unknown option: ',varargin{iarg}]);
    end
end

if strcmpi(mminmethod,'ks')
    % Set the K-S test significance number and find the significance level to use
    signum=[1.07 1.14 1.22 1.36 1.63];
    sigval=[0.2 0.15 0.1 0.05 0.01];
    sig=signum(E==sigval);
    if isempty(sig)
        error('BValue:UnknownOption',...
            ['E = ',num2str(E),'. It must be 0.02, 0.01, 0.05 or 0.01 only'])
    end
end


% Sort event magnitudes
Ms=sort(M);


if rounded
    if ~ibin
        A = diff(Ms);
        dbm = min(A(diff(Ms)~=0));
        bins = min(M)-dbm:dbm:max(M)+dbm;
    end
end
[nall,xall]=hist(M,bins);
pcdf=cumulative_magnitude(nall,0);


% Loop over increasing cut-off magnitude, find minimum magnitude at which
% straight line fit is accepted
if strcmpi(mminmethod,'ks') || strcmpi(mminmethod,'ww')
	for i=1:length(bins)    
        xc=xall(i:end);
        Mmin=min(xc);
        Mc=M(M>=Mmin);
    
        if length(Mc) < Nmin
           % warning('BValue:NoStraightLine',['Minimum number of events, ',...
           %     num2str(Nmin),' reached without finding a suitable distribution'])
            ierror = 1;
            Mmin = NaN;
            break
        end  
        
        db = min(diff(xc));
        
        switch fit
            case 'ml'
                fMinMag  = min(Mc);
                fMeanMag = mean(Mc);
                b = (1/(fMeanMag-(fMinMag-(db/2))))*log10(exp(1));
                a = log10(length(Mc)) + b * fMinMag;
                
            case 'log'
                [n,~]=hist(Mc,xc);
                cn=cumulative_magnitude(n,0);
                cn=cn';
    
                % Fit slope in log-lin space
                logcn = log10(cn);
                p = polyfit(xc,logcn,1);
                b = -p(1);
                a = p(2);
                
            otherwise
                error('BValue:UnknownOption',...
                    ['Unknown option: ',fit]);
        end
        
        % Test reconstructed values to determine if Mmin is reached
        switch mminmethod
            case 'ks'   % Kolomogorov-Smirnov
                nrecon=10.^(-b*xc);
                nrecon=nrecon/max(nrecon);
    
                [n,~]=hist(Mc,xc);
                cn=cumulative_magnitude(n,0);
                P=cn/max(cn);
                                
                % Check that P and nrecon have same size
                if size(P,2) > size(P,1)
                    P=P';
                end
                if size(nrecon,2) > size(nrecon,1)
                    nrecon=nrecon';
                end

                % Test differences between observed and reconstructed profile
                d=max(abs(P - nrecon));
                sig_n=sig/sqrt(length(Mc));
        
                % If d fits the profile, accept the results finish loop
                if d < sig_n
                    break
                end
                
            case 'ww'   % Wyss and Wiemer method
                nrecon=10.^(a-b*xc);  % Reconstruct b-value curve
                [n,~]=hist(Mc,xc);   % Data curve for cut data
                cn=cumulative_magnitude(n);  % Cumulative data curve

                % Check that P and nrecon have same size
                if size(cn,2) > size(cn,1)
                    cn=cn';
                end
                if size(nrecon,2) > size(nrecon,1)
                    nrecon=nrecon';
                end
                
                mf=sum(abs(cn-nrecon))/sum(cn);  % Misfit a la W&W (2000)
                
                if mf < E   % Take result when misfit first falls within limit
                    break
                end
            otherwise
                error('BValue:UnknownOption',...
                    ['Unknown option: ',mminmethod]);
        end    
 
	end
elseif ~isstring(mminmethod)
    Mc=M(M>=mminmethod);
   
    if length(Mc) < Nmin
      % 	warning('BValue:NoStraightLine',['Minimum number of events, ',...
      %  	num2str(Nmin),' reached without finding a suitable distribution'])
    	ierror = 1;
    else
     
        Mmin = min(Mc);
        xc = bins(bins>=Mmin);  
	
        db = min(diff(xc));
    
        switch fit
            case 'ml'
                fMinMag  = min(Mc);
                fMeanMag = mean(Mc);
                b = (1/(fMeanMag-(fMinMag-(db/2))))*log10(exp(1));
                a = log10(length(Mc)) + b * fMinMag;
                
            case 'log'
                [n,~]=hist(Mc,xc);
                cn=cumulative_magnitude(n,0);
                cn=cn';
    
                % Fit slope in log-lin space
                logcn = log10(cn);
                p = polyfit(xc,logcn,1);
                b = -p(1);
                a = p(2);
                
            otherwise
                error('BValue:UnknownOption',...
                    ['Unknown option: ',fit]);
        end
        [n,~]=hist(Mc,xc);   % Data curve for cut data
        cn=cumulative_magnitude(n);  % Cumulative data curve

        % Check that P and nrecon have same size
        if size(cn,2) > size(cn,1)
            cn=cn';
        end
    end
end
   
% If an error has been encountered (not enough events, or no straight line
% fit) set values to NaNs, make a plot and exit
if ierror == 1;  
    if fig
        hold on
        box on
        set(gca,'YScale','log','XScale','lin')
        plot(xall,pcdf,'go-')
        xlabel('Magnitude')

    end
    b=NaN; a=NaN; berr=[NaN NaN]; Mmin=NaN; mcdf=NaN; ecdf=NaN;

    return
end    

mcdf = 10.^(a-b*bins);
if fig
    hold on
    box on
    set(gca,'YScale','log','XScale','lin')
    plot(xall,pcdf,'go-')
    plot(xall(bins>=Mmin),pcdf(bins>=Mmin),'bo-')
    %plot(xall(bins>=Mmin),mcdf(bins>=Mmin),'r','LineWidth',2)
    plot(xall,mcdf,'r','LineWidth',2)
    xlabel('Magnitude')
end
    
% If error bounds have been requested
if isfinite(err)
    db = 0.01;
    
    if strcmpi(mminmethod,'ks')
        sig = signum(err==sigval);
        sig_n = sig/sqrt(length(Mc));
    end
    
    % Observed profile
    P=cn/max(cn);
    if size(P,2) > size(P,1)
        P=P';
    end
    if size(cn,2) > size(cn,1)
        cn=cn';
    end
        
    % Find upper limit
    iexit=0;
    tb=b;
    while iexit == 0;
        tb = tb + db;
             
        % Create reconstructed profile
        nrecon=10.^(a-tb*xc);
        if size(nrecon,2) > size(nrecon,1)
        	nrecon=nrecon';
        end
        
              
        switch mminmethod
            case 'ks'
                nrecon=nrecon/max(nrecon);
    
                % Test differences between observed and reconstructed profile
                d=max(abs(P - nrecon));

                % If d doesn't fit the profile, accept the results finish loop
                if d > sig_n
                    berr(1)=tb;
                    break
                end  
            
            case 'ww'
                mf=sum(abs(cn-nrecon))/sum(cn);
                if mf > err
                    berr(1) = tb;
                    break
                end
        end
    end
    
    % Find lower limit
    iexit=0;
    tb=b;
    while iexit == 0;
        tb = tb - db;
                
        % Create reconstructed profile
        nrecon=10.^(a-tb*xc);
        if size(nrecon,2) > size(nrecon,1)
        	nrecon=nrecon';
        end
        
        switch mminmethod
            case 'ks'
                nrecon=nrecon/max(nrecon);
    
                % Test differences between observed and reconstructed profile
                d=max(abs(P - nrecon));

                % If d doesn't fit the profile, accept the results finish loop
                if d > sig_n
                    berr(2)=tb;
                    break
                end  
            
            case 'ww'
                mf=sum(abs(cn-nrecon))/sum(cn);
                if mf > err
                    berr(2) = tb;
                    break
                end
        end
    end
    
    ae = log10(length(Mc)) + berr * min(Mc); % Use max likelihood for a errors 
    ecdf(1:length(bins),1) = 10.^(ae(1) - berr(1).*bins);
    ecdf(1:length(bins),2) = 10.^(ae(2) - berr(2).*bins);
    
    if fig
        plot(xall(bins>=Mmin),ecdf(bins>=Mmin,1),'r--')
        plot(xall(bins>=Mmin),ecdf(bins>=Mmin,2),'r--')
    end
else
    berr = NaN;
end








