!***************************************************************************************************
subroutine nearestneighbour(nafile,lnaf,nada,lnada,inumber)
!---------------------------------------------------------------------------------------------------
!
!   Main subroutine NA  - sampling a parameter space using a Neighbourhood algorithm
!
!   Input files:
!          nafile (length lnaf) - listing search parameters in keyword format
!
!   Output files:
!          nada (length lnada) - direct access file (NAD) of models
!
!   Comments:
!       The NAD file is a direct access compact format file containing all models generated by the
!       neighbourhood algorithm. (NAD files can be read in by multi-dimensional integration program 
!       NA-Bayesian)
!
!   Written by M. Sambridge, (RSES, ANU)
!               Last revision: J.P. Verdon, September 2011. 
!
!---------------------------------------------------------------------------------------------------
! Modifications:
!  James Wookey (University of Leeds), August 2004
!  James Verdon (University of Bristol), August 2011
!
!  JW 2004: Added parameters to main subroutine to suppress output, and change file name of output NA
!           ensemble
!  JW 2004: The f77 compatibility and MPI control have been stripped out
!  JV 2011: User specified output controls added (input and output files). If no output file 
!           specified, no nad is written. 
!  JV 2011: Input subroutine has been altered to a keyword driven format
!  JV 2011: Unneccesary options from input file have been hardwired
!              - Algorithm is hardwired to NA (rather than MCMC)
!              - Quasi random numbers is set to avoid the sobol stuff
!	 	       - Initial sampling is random (no a priori nad file available)
!	 		   - All random number work is now done by Wookey's RandomNumbers module
!	 		   - All outputs to screen and summary files have been suppressed
!  JV 2011: Much of the 'dead wood' has been stripped, alloowing the wood to be seen for the trees
!  JV 2011: Bug caused by swapping of na_models (sometimes called as bp) between 1D and 2D arrays
!           has been sorted out)
!  JV 2011: Dynamic memory allocation has been removed due to suspected bugs
!  JV 2011: inumber parameter included to define which of multiple runs the program is in
!---------------------------------------------------------------------------------------------------
!
!***************************************************************************************************

use na_globals

implicit none

integer 				:: numberOfModels
real 					:: misfit(nmod_max) 
real 					:: sum(nsample_max)    
real  					:: na_models(nd_max*nmod_max) 
real  					:: range(2,nd_max)   
real 					:: ranget(2,nd_max)
real 					:: scales(nd_max+1) 
real  					:: na_model(nd_max) 
real  					:: xcur(nd_max)  
real					:: work_NA1(nmod_max) 
real					:: work_NA2(nmod_max) 
real          			:: misfitval,mfitmin,mfitmean,mfitminc
real					:: model_opt(nd_max)

integer 				:: mfitord(nsample_max)
integer 				:: iwork_NA1(nmod_max) 
integer 				:: iwork_NA2(nsample_max) 
integer 				:: nd,nsamplei,nsample,nsleep,ncells,itmax,nclean
integer					:: ns,ntot,nh_user,nh
integer					:: i,ii,it,j,jj,mopt
integer         		:: nproc, iproc, ierr,lu_na,lu_nad
integer		   			:: lnaf,inumber,lnada
        
character       		:: header(nh_max)
character*256		  	:: nafile,nada
character*256   		:: fnme

logical					:: restartNA

common /NA_IO/lu_na,lu_nad
common /NAMPI/iproc,nproc

lu_na = 30
lu_nad = 34

! set up variables and start MPI
iproc = 0
nproc = 1

! User specific setup for forward modelling
write(6,*)'     --> Initialising ranges'
call user_init(nd,range,scales)       
if(nd.le.0)then
            write(6,*)
            write(6,*)' Error in user supplied routine user_init'
            write(6,*)
            write(6,*)' Number of dimensions in parameter space '
            write(6,*)' is less than or equal to zero'
            write(6,*)' This must be an error in the users code'
            write(6,*)
            write(6,*)' Remedy: fix the code '
            write(6,*)
            stop
end if
if(nd.gt.nd_max)then
            write(6,*)
            write(6,*)' Sorry, NA cant currently handle that'
            write(6,*)' many dimensions.'
            
            write(6,*)' Number of dimensions in parameter space '
            write(6,*)' has an upper limit of ',nd_max
            write(6,*)
            stop
end if

! Open and read NA input files
open(lu_na,file=nafile(1:lnaf),status='old')
if(inumber.eq.1)write(6,*)'     --> Reading NA options from ',nafile(1:lnaf)
call NA_options(nd, nsamplei, nsample, nsleep, ncells, itmax, nclean)
close(lu_na)

! Initialize NA routines.
write(6,*)'     --> Initialising NA routines'
call NA_initialize(range,ranget,scales,nd,xcur,nsample,ncells,restartNA)

! Generate or read in starting models 
call NA_initial_sample(na_models,nd,ranget,range,nsamplei,numberOfModels,scales,misfit)

! MAIN OPTIMIZATION LOOP
ntot = 0
ns = nsamplei
write(6,*)'     --> Searching parameter space'
do it = 1,itmax+1
	! Calculate misfit values for each model in the current population.
    do i = iproc+1,ns,nproc

		! Decode current model and put into array model.
		ii = 1+(i-1+ntot)*nd

    	call transform2raw(na_models(ii),nd,range,scales,na_model)
    	misfitval = 0.0

		! Generate the forward model and compute misfit
    	call forward(nd, na_model, misfitval,ntot+i,inumber)

        jj = ntot + i
        misfit(jj) = misfitval

 	enddo
    
! Calculate properties of current misfit distribution. (Mean,min,best model etc.)
	call NA_misfits(misfit,ns,it,ntot,mfitmin,mfitminc,mfitmean,mopt,ncells,work_NA2,  &
           iwork_NA1,iwork_NA2,mfitord)

! copy optimum model from current population, to array model_opt.
    ii = 1 + (mopt-1)*nd
    call transform2raw(na_models(ii),nd,range,scales,na_model)

    do j=1,nd
		model_opt(j) = na_model(j)
    end do

    ntot = ntot + ns
    ns = nsample

    if(it.eq.itmax+1)exit

! Generate a new sample using Neighbourhood algorithm (resample version)
	call NA_sample(na_models, ntot, nsample, nd, nsleep,ncells, misfit, mfitord, ranget,   &
               xcur, restartNA, nclean, work_NA1)

enddo
! FINISHED OPTIMIZATION LOOP
!if (inumber.eq.3.and.itype)write(6,*)'     --> Best fit model found'

! transform all models back to scaled units
do i=1,ntot
	ii = 1 + (i-1)*nd
    call transform2raw(na_models(ii),nd,range,scales,na_models(ii))
enddo

nh = 1

! Write out ensemble of models as a direct access file (if asked)
if (lnada.gt.0)then
	write(6,*)'     --> Writing model ensemble to ',nada(1:lnada)
	fnme=nada(1:lnada)
	call NA_header(lu_nad,fnme,header,nh,nd,range,scales,nsamplei,nsample,ncells,nh_user)                
	call write_nad(lu_nad,fnme,nd,ntot,nh,nh_user,header,na_models,misfit)   
	close(lu_nad)
endif

!if (inumber.eq.3)write(6,*)'     --> NA inversion complete - best fit location found'

return
end subroutine nearestneighbour
!***************************************************************************************************


!***************************************************************************************************
subroutine NA_initialize(range,ranget,scales,nd,x,nsample,ncells,restartNA)
!---------------------------------------------------------------------------------------------------
!    
!       NA_initialize - performs minor initialization tasks for NA algorithm.
!       Calls no other routines.
!                  M. Sambridge, Oct. 1996
!***************************************************************************************************

use na_globals
use RandomNumbers

implicit none

real            :: range(2,nd_max)
real            :: ranget(2,nd_max)
real            :: scales(nd_max+1)
real            :: x(nd_max)
real            :: rval(2)
real			:: a
integer 		:: nd, nsample, ncells
integer 		:: i,iproc,nproc,ic,idnext
logical         :: restartNA


common /NA_init/idnext,ic

!  set logical switch for first call to NA_sample (ensures distance list is initialized)
restartNA = .true.

! set initial parameter for NA walk
ic = 1

! Initialize pseudo-random number generator - now using RandomNumbers module
a=RandomR4()
!iseed0 = iseed
!a = ran3(iseed)

! Normalize parameter ranges by a-priori model co-variances
if(scales(1).eq.0.0)then
	! First option: No transform (All a priori model co-variances are equal to unity)
	do i=1,nd
		ranget(1,i) = range(1,i)
        ranget(2,i) = range(2,i)
        scales(i+1) = 1.0
    enddo

elseif(scales(1).eq.-1.0)then
	! Second option: Use parameter range as a priori model co-variances 
    do i=1,nd
    	ranget(1,i) = 0.0
        ranget(2,i) = 1.0
        scales(i+1) = range(2,i)-range(1,i)
    enddo
else
	! Third option: Use scales array as a priori model co-variances 
	do i=1,nd
    	if(scales(i+1).eq.0.0)then
        	write(*,200)i
            call na_abort
        end if
        ranget(1,i)  = 0.0
        ranget(2,i)  = (range(2,i)-range(1,i))/scales(i+1)
    enddo
endif

! calculate axis increments and initialize current point (used by NA_sample) to mid-point of parameter space
do i=1,nd
	x(i) = (ranget(2,i)+ranget(1,i))/2.0
enddo
        
 200    format(/' Error in subroutine NA_initialize '//,  &
          ' Input a priori model co-variance is equal to zero',  &
          ' for parameter ',i4/  &
          ' This is not valid'//  &
          ' Remedy - check and adjust input',  &
          ' a priori co-variances'/)

return
end subroutine NA_initialize
!***************************************************************************************************


!***************************************************************************************************
subroutine NA_initial_sample(na_models,nd,range,rangeo,nsample,numberOfModels,&
    scales,misfit)
!---------------------------------------------------------------------------------------------------
!    
!       NA_initial_sample - generates initial sample for NA algorithm.
!
!   Comments:
!
!      Assumes n-dimensional Sobol sequence has been initialized
!      Will generate a minimum of two samples.
!
!      Assumes ran3 has been initialized.
!
!       Calls no other routines.
!
!                  M. Sambridge, Oct. 1996
!                  (Updated for ran3 Aug. 1997)
!***************************************************************************************************
  
use na_globals
use RandomNumbers

implicit none

real          		:: na_models(nd_max*nmod_max)
real          		:: range(2,nd_max)
real          		:: rangeo(2,nd_max)
real          		:: scales(nd_max+1)
real          		:: misfit(nmod_max)
real		  		:: a,b
integer 	  		:: i,j,nd, nsample, numberOfModels
integer		  		:: count
character     		:: header(nh_max)
character*256 		:: fnme

! Generate initial uniform random sample using a uniform random distribution
count=0
do i=1,nsample
	do j=1,nd
		count=count+1
		!a = ran3(iseed)
		a=RandomR4()
		b = 1-a
        na_models(count) = b*range(1,j) + a*range(2,j) ! Bug fix - na_models stays 1D
!        na_models(j,i) = b*range(1,j) + a*range(2,j)
        
    enddo
enddo

return
end subroutine NA_initial_sample
!***************************************************************************************************


!***************************************************************************************************
subroutine NA_sample(na_models1d, ntot, nsample, nd, nsleep, ncells,misfit, mfitord, range,   &
 xcur, restartNA, nclean, dlist)
!---------------------------------------------------------------------------------------------------
!    
!       NA_sample - generates a new sample of models using 
!                   the Neighbourhood algorithm by distributing
!          nsample new models in ncells cells.
!
!   Comments:
!       If xcur is changed between calls then restartNA 
!       must be set to true. logical restartNA must also 
!       be set to true on the first call.
!
!       Calls are made to various NA_routines.
!
!                  M. Sambridge
!                  Last updated Sept. 1999.
!***************************************************************************************************

use RandomNumbers  
use na_globals

implicit none

real            	:: na_models1d(nd_max*nmod_max)
real				:: na_models(nd_max,nmod_max)
real           		:: range(2,nd_max)
real           		:: misfit(nmod_max)
real           		:: xcur(nd_max)
real           		:: dlist(nmod_max)
real           		:: xdum(100)
real				:: dcount,dsum,mopt,dminx,x1,x2
integer         	:: nsample, nd, nsleep
integer        		:: mfitord(nsample_max)
integer        		:: cell
integer				:: ntot,ncells,nclean,ind_cellnext,icount,ind_celllast,ind_cell
integer				:: i,j,il,is,iw,kd,id,ic,idnext,idiff,nrem
integer				:: nsampercell,nodex,count
logical         	:: resetlist
logical         	:: restartNA

common /NA_init/idnext,ic
save id

! Bug fix - convert na_models into 2-dimensional array
count=0
do i=1,nmod_max
	do j=1,nd
		count=count+1
		na_models(j,i)=na_models1d(count)
	enddo
enddo

! choose initial axis randomly
idnext=RandomInt(1,nd)

! initialize some variables
ic = ic + 1
if(mod(ic,nclean).eq.0)resetlist = .true.
idiff = 0
cell = 1
mopt = mfitord(cell)
ind_cellnext = mopt
ind_celllast = 0
dsum = 0.0
dcount = 0.0
nrem = mod(nsample,ncells)
if(nrem.eq.0)then
	nsampercell = nsample/ncells
else
    nsampercell = 1+nsample/ncells
endif

icount = 0

! loop over samples
do is = 1,nsample
	! choose Voronoi cell for sampling
    ind_cell = ind_cellnext 
    icount = icount + 1
    if(ind_cell.ne.ind_celllast)then
		! reset walk to chosen model
    	call NA_restart(na_models,nd,ind_cell,xcur,restartNA)
	endif

	if(restartNA)then
		resetlist = .true.
    	restartNA = .false.
	endif

	!  loop over walk steps
    do il = 1,nsleep
    	do iw = 1,nd
			
			! update dlist and nodex for new axis
			if(.not.resetlist)then
				! incremental update
        		call NNupdate_dlist(idnext,id,dlist,na_models,nd,ntot,xcur,nodex,dminx)
    		else				
				! full update
        		call NNcalc_dlist(idnext,dlist,na_models,nd,ntot,xcur,nodex,dminx)
    		endif

			id = idnext

			! Calculate intersection of current Voronoi cell with current 1-D axis
    		call NNaxis_intersect(xcur,id,dlist,na_models,nd,ntot,nodex,range(1,id),range(2,id), &
    		x1,x2)
			! Generate new node in Voronoi cell of input point
    		kd = id + (cell-1)*nd
    		call NA_deviate (x1,x2,kd,xcur(id))
			! increment axis 
    		idnext = idnext + 1
    		if(idnext.gt.nd)idnext=1
		enddo
    enddo
	
	! put new sample in list
    j = ntot+is
    do i=1,nd
    	na_models(i,j) = xcur(i)
    enddo
	ind_celllast = ind_cell

 	if(icount.eq.nsampercell)then
        icount = 0 
        cell = cell + 1
        ind_cellnext = mfitord(cell)
        if(cell.eq.nrem+1)nsampercell = nsampercell - 1
    endif
enddo

! Bug fix - convert na_models back into 1-dimensional array
count=0
do i=1,nmod_max
	do j=1,nd
		count=count+1
		na_models1d(count)=na_models(j,i)
	enddo
enddo

return
end subroutine NA_sample
!***************************************************************************************************


!***************************************************************************************************
subroutine NNaxis_intersect(x,dim,dlist,bp,nd,nb,nodex,xmin,xmax,x1,x2)
!---------------------------------------------------------------------------------------------------
!    
!   NNaxis_intersect - find intersections of current Voronoi cell 
!            with current 1-D axis.
!
!       Input:
!         x(nd)      :point on axis
!         dim      :dimension index (defines axis)
!         dlist      :set of distances of base points to axis 
!         bp(nd,nb)      :set of base points
!         nd      :number of dimensions
!         nb      :number of base points 
!         resetlist      :TRUE if dlist and nodex is to be calculated
!         nodex      :index of base node closest to x
!         dmin_in      :distance of base node closest to x
!         xmin      :start point along axis
!         xmax      :end point along axis
!
!       Output:
!         x1      :intersection of first Voronoi boundary 
!         x2      :intersection of second Voronoi boundary 
!
!       Comment:
!           This method uses a simple formula to exactly calculate
!      the intersections of the Voronoi cells with the 1-D axis.
!      It makes use of the perpendicluar distances of all nodes
!      to the current axis contained in the array dlist. 
!
!           The method involves a loop over ensemble nodes for 
!      each new intersection found. For an axis intersected
!      by ni Voronoi cells the run time is proportional to ni*ne.
!
!      It is assumed that the input point x(nd) lies in
!      the Vcell of nodex, i.e. nodex is the closest node to x(nd).
!
!      Note: If the intersection points are outside of either
!            axis range then the axis range is returned, i.e.
!
!                  x1 is set to max(x1,xmin) and  
!                  x2 is set to min(x2,xmin) and  
!
!                                       M. Sambridge, RSES, June 1998
!***************************************************************************************************

use na_globals

implicit none

real			:: x(nd_max)
real          	:: bp(nd_max,nmod_max)
real          	:: dlist(nmod_max)
real			:: xmin,xmax,x1,x2,dx,dpc,dp0,x0,xc,xi
integer         :: dim
integer			:: nd,nb,nodex,j

! search through nodes
x1 = xmin
x2 = xmax
dp0   = dlist(nodex)
x0    = bp(dim,nodex)

! find intersection of current Voronoi cell with 1-D axis
do j=1,nodex-1
	xc    = bp(dim,j)
    dpc   = dlist(j)
	! calculate intersection of interface (between nodes nodex and j) and 1-D axis.
    dx = x0 - xc
    if(dx.ne.0.0)then
    	xi = 0.5*(x0+xc+(dp0-dpc)/dx)
        if(xi.gt.xmin.and.xi.lt.xmax)then
        	if(xi.gt.x1.and.x0.gt.xc)then
            	x1 = xi
            else if(xi.lt.x2.and.x0.lt.xc)then
                x2 = xi
            endif
        endif
    endif
enddo

do j=nodex+1,nb
	xc    = bp(dim,j)
	dpc   = dlist(j)
	! calculate intersection of interface (between nodes nodex and j) and 1-D axis.
    dx = x0 - xc
    if(dx.ne.0.0)then
        xi = 0.5*(x0+xc+(dp0-dpc)/dx)
        if(xi.gt.xmin.and.xi.lt.xmax)then
            if(xi.gt.x1.and.x0.gt.xc)then
                x1 = xi
            elseif(xi.lt.x2.and.x0.lt.xc)then
                x2 = xi
            endif
        endif
    endif
enddo

return
end subroutine NNaxis_intersect
!***************************************************************************************************


!***************************************************************************************************
subroutine NNcalc_dlist(dim,dlist,bp,nd,nb,x,nodex,dminx)
!---------------------------------------------------------------------------------------------------
!    
!       Subroutine NNcalc_dlist - calculates square of distance from
!                                 all base points to new axis (defined
!                                 by dimension dim through point x.
!                                 It also updates the nearest node and
!                                 distance to the point x.
!
!       This is a full update of dlist, i.e. not using a previous dlist.
!***************************************************************************************************

use na_globals

implicit none

real			:: bp(nd_max,nmod_max)
real          	:: x(nd_max)
real          	:: dlist(nmod_max)
real			:: dminx,d,dsum,dmin,dnodex
integer       	:: dim
integer			:: nd,nb,nodex,i,j

dmin = 0.
do j=1,dim-1
    d = (x(j)-bp(j,1))
    d = d*d
    dmin = dmin + d
enddo
do j=dim+1,nd
    d = (x(j)-bp(j,1))
    d = d*d
    dmin = dmin + d
enddo
dlist(1) = dmin
d = (x(dim)-bp(dim,1))
d = d*d
dmin = dmin + d
nodex = 1

do i=2,nb
    dsum = 0.
    do j=1,dim-1
        d = (x(j)-bp(j,i))
        d = d*d
        dsum = dsum + d
    end do
    do j=dim+1,nd
        d = (x(j)-bp(j,i))
        d = d*d
        dsum = dsum + d
    end do
    dlist(i) = dsum
    d = (x(dim)-bp(dim,i))
    d = d*d
    dsum = dsum + d
    if(dmin.gt.dsum)then
        dmin = dsum
        nodex = i
    end if
    dnodex = dmin
end do

return
end subroutine NNcalc_dlist
!***************************************************************************************************


!***************************************************************************************************
subroutine NNupdate_dlist(dim,dimlast,dlist,bp,nd,nb,x,node,dmin)
!---------------------------------------------------------------------------------------------------
!    
!   Subroutine NNupdate_dlist - calculates square of distance from 
!                 all base points to new axis, assuming
!                                    dlist contains square of all distances 
!                 to previous axis dimlast. It also
!                 updates the nearest node to the
!                 point x through which the axes pass.
!***************************************************************************************************

use na_globals

implicit none

real        	:: bp(nd_max,nmod_max)
real          	:: x(nd_max)
real          	:: dlist(nmod_max)
real			:: dmin,d1,d2,ds
integer       	:: dim,dimlast
integer			:: i,nd,nb,node

d1 = (x(dimlast)-bp(dimlast,1))
d1 = d1*d1
dmin = dlist(1)+d1
node = 1
d2 = (x(dim)-bp(dim,1))
d2 = d2*d2
dlist(1) = dmin-d2
do i=2,nb
    d1 = (x(dimlast)-bp(dimlast,i))
    ds = d1
    d1 = dlist(i)+d1*d1
    if(dmin.gt.d1)then
        dmin = d1
        node = i
    end if
    d2 = (x(dim)-bp(dim,i))
    d2 = d2*d2
    dlist(i) = d1-d2
enddo

return
end subroutine NNupdate_dlist
!***************************************************************************************************


!***************************************************************************************************
subroutine NA_misfits(misfit,nsample,it,ntot,mfitmin,mfitminc,mfitmean,mopt,ncells,work,ind,&
  iwork,mfitord)
!---------------------------------------------------------------------------------------------------
!    
!       NA_misfits - calculate performance statistics for NA algorithm.
!
!       Calls no other routines.
!
!                  M. Sambridge, Oct. 1996
!***************************************************************************************************

use na_globals

implicit none

real           	:: misfit(nmod_max)
real            :: mfitmin
real            :: mfitmean
real            :: mfitminc
real            :: work(nmod_max)
real			:: flow,na_select

integer         :: ind(nmod_max)
integer         :: mfitord(nsample_max)
integer         :: iwork(nsample_max)
integer			:: nsample,it,ntot,mopt,ncells,iopt
integer			:: i,j,ntotal,iselect

mfitminc = misfit(ntot+1)
mfitmean = mfitminc
iopt = ntot+1

do i=ntot+2,ntot+nsample
	mfitmean = mfitmean + misfit(i)
	if(misfit(i).lt.mfitminc)then
    	mfitminc = misfit(i)
        iopt = i
    endif 
enddo
mfitmean = mfitmean/real(nsample)

if(mfitminc.lt.mfitmin.or.it.eq.1)then
	mopt = iopt
    mfitmin = mfitminc
endif

! find models with lowest ncells misfit values
if(ncells.eq.1)then
	mfitord(1) = mopt
else 
	ntotal = ntot+nsample
	do i=1,ntotal
		ind(i) = i
        work(i) = misfit(i)
    enddo
   
	! jumble initial indices to randomize order of models when misfits are equal
    call jumble(ind,work,ntotal)
	flow = na_select(ncells,ntotal,work,ind,iselect)

	do j=1,ncells
		iwork(j) = ind(j)
	enddo 
	
	! order misfit of lowest ncells
    call indexx(ncells,work,ind)
	
	do j=1,ncells
		mfitord(j) = iwork(ind(j))
    enddo
    
endif

return
end subroutine NA_misfits
!***************************************************************************************************


!***************************************************************************************************
subroutine NA_deviate(x1,x2,i,deviate)
!---------------------------------------------------------------------------------------------------
!    
!     NA_deviate - generates a random deviate according to
!                  a given distribution using a 1-D SAS sequence.
!         or a pseudo-random sequence depending of logical
!         `sobol'
!
!     Comments:
!      If sobol = .true.: (Quasi-random number)
!
!         This routine generates a random number 
!         between x1 and x2.
!         The parameter i is the sequence number from 
!         which the quasi random devaite is drawn.
!
!      If sobol = .false. (Pseduo-random number)
!
!         ran3 is called to calculate a deviate which is
!         scaled to the input boundaries x1,x2.
!
!      This version is for resample mode and simply generates
!      a deviate between input values x1 and x2.
!***************************************************************************************************

use na_globals
use RandomNumbers  

implicit none

real			:: x1,x2,deviate,ran
integer			:: i

!ran = ran3(iseed)  ! Switch to use RandomNumbers module instead
ran=RandomR4()
deviate = x1 + (x2-x1)*ran

return
end subroutine NA_deviate
!***************************************************************************************************


!***************************************************************************************************
subroutine NA_restart(na_models,nd,mreset,x,restartNA)
!---------------------------------------------------------------------------------------------------
!    
!       NA_restart - resets NA walk to start from input model.
!
!       Calls no other routines.
!
!                  M. Sambridge, Oct. 1996
!***************************************************************************************************

use na_globals

implicit none

real			:: na_models(nd_max,nmod_max)
real			:: x(nd_max)
integer			:: i,nd,mreset
logical			:: restartNA

do i=1,nd
	x(i) = na_models(i,mreset)
enddo

restartNA = .true.

return
end subroutine NA_restart
!***************************************************************************************************


!***************************************************************************************************
subroutine NA_header(lu,fnme,header,nh,nd,range,scales,n1,n2,n3,nhu)
!---------------------------------------------------------------------------------------------------
!
!       NA_header - writes NA-specific information to NAD header.
!
!          This routine adds various NA-header info to
!          the header written by the user.
!
!       Calls no other routines.
!
!                  M. Sambridge, June 1999
!***************************************************************************************************

use na_globals

implicit none

real				:: range(2,nd_max)
real            	:: scales(nd_max+1)
real				:: hu,rlen
integer				:: lu,nh,nd,n1,n2,n3,nhu,len,nh_na,nh_tot
character         	:: header(nh_max)
character*256     	:: fnme

! calculate total header length
rlen = 3*nd + 4
len = 4*rlen+nh
hu = nh
nh_na   = 4*rlen
nh_tot  = len

if(nh_tot.gt.nh_max)then
    write(*,*)
    write(*,*)' Error - header array too small'
    write(*,*)
    write(*,*)'         current size = ',nh_max
    write(*,*)'        required size = ',nh_tot
    write(*,*)
    write(*,*)' Remedy - adjust nh_max in parameter',  &
                        ' file and recompile'
    write(*,*)
    call na_abort
endif

! write out header information
call write_header(lu,fnme,len,nd,nh,range,scales,n1,n2,n3,header)

nh = nh_tot

! read header information into character string
call read_header(lu,fnme,nh,len,header)

return
end subroutine NA_header
!***************************************************************************************************


!***************************************************************************************************
subroutine write_header(lu,fnme,len,nd,nh,range,scales,n1,n2,n3,header)
!---------------------------------------------------------------------------------------------------
!
!       write_header - converts header information into a character
!             string by writing it to a direct access file
!             and then reading it back as a character string
!
!       Calls no other routines.
!
!                  M. Sambridge, June 1999
!***************************************************************************************************

use na_globals

implicit none

real            	:: range(2,nd_max)
real            	:: scales(nd_max+1)
integer				:: lu,len,nd,nh,n1,n2,n3
character         	:: header(nh_max)
character*256     	:: fnme

open(lu,file=fnme,status='unknown',form='unformatted',access='direct',recl=len)
write(lu,rec=1)n1,n2,n3,range(:,1:nd),scales(1:nd+1),header(1:nh)

close(lu)

return
end subroutine write_header
!***************************************************************************************************


!***************************************************************************************************
subroutine read_header(lu,fnme,nh,len,header)
!---------------------------------------------------------------------------------------------------
!
!       read_header - converts header information into a character
!            string by writing it to a direct access file
!            and then reading it back as a character string
!
!       Calls no other routines.
!
!                  M. Sambridge, June 1999
!***************************************************************************************************

use na_globals

implicit none

integer				:: lu,nh,len
character         	:: header(nh_max)
character*256     	:: fnme

open(lu,file=fnme,status='unknown',form='unformatted',access='direct',recl=len)
read(lu,rec=1)header(1:nh)
close(lu)

return
end subroutine read_header
!***************************************************************************************************

!***************************************************************************************************
subroutine write_nad(lu,fnme,nd,ne,nh,nhu,header,models,data)
!---------------------------------------------------------------------------------------------------
!
!       write_nad - write a direct access file in multi-record NAD format
!
!       Input:
!             lu                : logical unit of file
!             fnme              : filename
!             nhmax             : maximum size of array header
!             ndmax             : maximum size of array data
!             nemax             : maximum size of array models
!             iform             : =0 then single record format
!                                 =1 then multi-record format (for large nd)
!
!       Output:
!             nh                : length in bytes of file header
!             nhu               : length in bytes of user portion of header
!             nd                : dimension of parameter space
!             ne                : number of models in ensemble
!             header            : header character string of length nh (char)
!             data(nd)          : array of data values for each model (real*4)
!             models(nd,ne)     : array of model values  (real*4)
!
!       Comments:
!                The direct access NAD file format:
!
!                VARIABLE       TYPE            SIZE IN BYTES
!                nd             int             4
!                ne             int             4
!                nh             int             4
!                nhu            int             4
!                header         character       nh
!                models         real*4          4*nd*ne
!                data           real*4          4*nd
!                tail           real*4          4
!
!                In single record mode a direct access file of size
!                [4x(4+nd*ne+ne+1) + nh] bytes is produced.
!
!                In multi record mode a direct access file of size
!                [(ne+1)*(max(20+nh,4(nd+1))] bytes is produced.
!
!               Calls are made to subroutine read_da.
!
!                This routine assumes that direct access
!                files are opened with the record length specified
!                in bytes. This is the default for most machines
!                but not alpha machines. (Often a compiler option is
!                available on the DEC/compaq to use bytes rather than
!                4-byte words.)
!
!                                       M. Sambridge, RSES, November 2001
!
!***************************************************************************************************

use na_globals

implicit none

real 				:: models(nd,ne)
real            	:: data(ne)
real            	:: tail

integer				:: lu,nd,ne,nh,nhu
integer				:: i,is1,mul,len1,is2,len2,lenh,num

character         	:: header(nh_max)
character*256     	:: fnme

! calculate length of header
len1 = 4*5+nh
len2 = 4*(nd+1)
mul  = 1 + (len1-1)/len2
lenh = mul*len2
num = ne + mul
is1 = num*len2
is2 = 4*(5+nd*ne+ne)+nh

! write header
open(lu,file=fnme,status='unknown',form='unformatted',access='direct',recl=lenh)

! write out header for multi-record format
write(lu,rec=1)-mul,nd,ne,nh,nhu,header(1:nh)
close(lu)
	
! write models
open(lu,file=fnme,status='unknown',form='unformatted',access='direct',recl=len2)
do i=1,ne
    call wnad(lu,mul+i,nd,models(1,i),data(i))
enddo
close(lu)

return
end subroutine write_nad
!---------------------------------------------------------------------------------------------------
subroutine wnad(lu,i,nd,models,data)

implicit none

real            models(nd)
real            data
integer 		:: lu,i,nd

write(lu,rec=i)models,data

return
end subroutine wnad
!***************************************************************************************************


!***************************************************************************************************
subroutine transform2raw(model_sca,nd,range,scales,model_raw)
!---------------------------------------------------------------------------------------------------
!    
!       transform2raw - transforms model from scaled to raw units.
!
!   Input:
!         nd      : dimension of parameter space
!         model_sca(nd)   : model in scaled co-ordinates
!         range(2,nd)   : min and max of parameter space 
!              in raw co-ordinates.
!         scales(nd+1)   : range scale factors
!
!   Output:
!         model_raw(nd)   : model in scaled co-ordinates
!
!   Comments:
!            This routine transforms a model in dimensionless scaled 
!       co-ordinates to input (raw) units.
!
!       Calls no other routines.
!
!                                               M. Sambridge, March 1998.
!***************************************************************************************************

use na_globals

implicit none

real         	:: model_raw(nd_max)
real          	:: model_sca(nd)
real          	:: scales(nd_max+1)
real          	:: range(2,nd_max)
real			:: a,b
integer			:: i,nd

if(scales(1).eq.0.0)then
	do i=1,nd
		model_raw(i) = model_sca(i)
	enddo
else if(scales(1).eq.-1.0)then
	do i=1,nd
		b = model_sca(i)
		a = 1-b
		model_raw(i) = a*range(1,i) + b*range(2,i)
	enddo
else
	do i=1,nd
		model_raw(i) = range(1,i) + scales(i+1)*model_sca(i)
	enddo
endif

return
end subroutine transform2raw
!***************************************************************************************************


!***************************************************************************************************
subroutine jumble(iarr,arr,n)
!---------------------------------------------------------------------------------------------------
!    
!       jumble - randomly re-arranges input array  
!
!       Calls ran3 and assumes that it has been initialized.
!
!                                               M. Sambridge, Oct. 1999.
!***************************************************************************************************

use RandomNumbers

implicit none

real	    :: arr(n)
real		:: val,rn
integer		:: n,ival,j,k
integer     :: iarr(n)
        
rn = n
do j=1,n
    !val = ran3(iseed)
    val=RandomR4()
    k = 1 + int(val*rn)
    if(k.eq.n+1)then
        k = n
    else if(k.gt.n)then
        write(6,*)' error in jumble k',k,' val',val,' rn',rn
        call na_abort
    end if
    ival = iarr(j)
    iarr(j) = iarr(k)
    iarr(k) = ival
    val = arr(j)
    arr(j) = arr(k)
    arr(k) = val
enddo

return
end subroutine jumble
!***************************************************************************************************


!***************************************************************************************************
subroutine na_abort
!---------------------------------------------------------------------------------------------------
!    
!       na_abort - randomly re-arranges input array 
!				   Used in place of a stop to ensure that all processes are stopped in MPI mode
!***************************************************************************************************

implicit none

stop 'NA ended unexpectedly'
end
!***************************************************************************************************


!***************************************************************************************************
! NUMERICAL RECIPES SUBROUTINES - DO NOT ALTER
!***************************************************************************************************
!
!---------------------------------------------------------------------------------------------------
!
!   Numerical recipes routine
!
!---------------------------------------------------------------------------------------------------
!
      SUBROUTINE indexx(n,arr,indx)
      INTEGER n,indx(n),M,NSTACK
      REAL arr(n)
      PARAMETER (M=7,NSTACK=50)
      INTEGER i,indxt,ir,itemp,j,jstack,k,l,istack(NSTACK)
      REAL a
      do 11 j=1,n
        indx(j)=j
11    continue
      jstack=0
      l=1
      ir=n
1     if(ir-l.lt.M)then
        do 13 j=l+1,ir
          indxt=indx(j)
          a=arr(indxt)
          do 12 i=j-1,1,-1
            if(arr(indx(i)).le.a)goto 2
            indx(i+1)=indx(i)
12        continue
          i=0
2         indx(i+1)=indxt
13      continue
        if(jstack.eq.0)return
        ir=istack(jstack)
        l=istack(jstack-1)
        jstack=jstack-2
      else
        k=(l+ir)/2
        itemp=indx(k)
        indx(k)=indx(l+1)
        indx(l+1)=itemp
        if(arr(indx(l+1)).gt.arr(indx(ir)))then
          itemp=indx(l+1)
          indx(l+1)=indx(ir)
          indx(ir)=itemp
        endif
        if(arr(indx(l)).gt.arr(indx(ir)))then
          itemp=indx(l)
          indx(l)=indx(ir)
          indx(ir)=itemp
        endif
        if(arr(indx(l+1)).gt.arr(indx(l)))then
          itemp=indx(l+1)
          indx(l+1)=indx(l)
          indx(l)=itemp
        endif
        i=l+1
        j=ir
        indxt=indx(l)
        a=arr(indxt)
3       continue
          i=i+1
        if(arr(indx(i)).lt.a)goto 3
4       continue
          j=j-1
        if(arr(indx(j)).gt.a)goto 4
        if(j.lt.i)goto 5
        itemp=indx(i)
        indx(i)=indx(j)
        indx(j)=itemp
        goto 3
5       indx(l)=indx(j)
        indx(j)=indxt
        jstack=jstack+2
        if(jstack.gt.NSTACK)stop 'NSTACK too small in indexx'
        if(ir-i+1.ge.j-l)then
          istack(jstack)=ir
          istack(jstack-1)=i
          ir=j-1
        else
          istack(jstack)=j-1
          istack(jstack-1)=l
          l=i
        endif
      endif
      goto 1
      END
      
!
!---------------------------------------------------------------------------------------------------
!
!   Numerical recipes routine adapted to give ind and iselect
!
!---------------------------------------------------------------------------------------------------
!
      FUNCTION na_select(k,n,arr,ind,iselect)
      INTEGER k,n
      REAL na_select,arr(n)
      integer ind(n)
      INTEGER i,ir,j,l,mid
      REAL a,temp
      l=1
      ir=n
1     if(ir-l.le.1)then
        if(ir-l.eq.1)then
          if(arr(ir).lt.arr(l))then
            temp=arr(l)
            arr(l)=arr(ir)
            arr(ir)=temp
            itemp=ind(l)
            ind(l)=ind(ir)
            ind(ir)=itemp
          endif
        endif
        na_select=arr(k)
        iselect=ind(k)
        return
      else
        mid=(l+ir)/2
        temp=arr(mid)
        arr(mid)=arr(l+1)
        arr(l+1)=temp
        itemp=ind(mid)
        ind(mid)=ind(l+1)
        ind(l+1)=itemp
        if(arr(l+1).gt.arr(ir))then
          temp=arr(l+1)
          arr(l+1)=arr(ir)
          arr(ir)=temp
          itemp=ind(l+1)
          ind(l+1)=ind(ir)
          ind(ir)=itemp
        endif
        if(arr(l).gt.arr(ir))then
          temp=arr(l)
          arr(l)=arr(ir)
          arr(ir)=temp
          itemp=ind(l)
          ind(l)=ind(ir)
          ind(ir)=itemp
        endif
        if(arr(l+1).gt.arr(l))then
          temp=arr(l+1)
          arr(l+1)=arr(l)
          arr(l)=temp
          itemp=ind(l+1)
          ind(l+1)=ind(l)
          ind(l)=itemp
        endif
        i=l+1
        j=ir
        a=arr(l)
        ia=ind(l)
3       continue
          i=i+1
        if(arr(i).lt.a)goto 3
4       continue
          j=j-1
        if(arr(j).gt.a)goto 4
        if(j.lt.i)goto 5
        temp=arr(i)
        arr(i)=arr(j)
        arr(j)=temp
        itemp=ind(i)
        ind(i)=ind(j)
        ind(j)=itemp
        goto 3
5       arr(l)=arr(j)
        arr(j)=a
        ind(l)=ind(j)
        ind(j)=ia
        if(j.ge.k)ir=j-1
        if(j.le.k)l=i
      endif
      goto 1
      END
!
