!***************************************************************************************************
subroutine input
!---------------------------------------------------------------------------------------------------
!
! Subroutine INPUT reads control data from specified input file
!
! Parameters to be read are:
!    rp = rock physics type (1-8)
!			1 = "vti" = vti only
!			2 = "verticalscalar" = single vertical scalar fracture
!			3 = "dippingscalar" = single dipping scalar fracture
!			4 = "doubleverticalscalar" = two vertical scalar fractures
!			5 = "vtiandverticalscalar" = vti and a single vertical scalar fracture
!			6 = "verticalznzt" = single vertical zn/zt fracture
!			7 = "dippingznzt = single dipping zn/zt fracture
!			8 = "vtiandverticalznzt" = vti and a single zn/zt fracture
!
!    itype = inversion type (1-4)
!			1 = "equal" = weight psi and dVs equally
!			2 = "psi" = using fast directions only
!			3 = "dVs" = using dVs only
!			4 = "weighted" = weighted misfit function, where MF = MF(psi) + W*MF(mag)
!
!    vpi,vsi,rhoi = background properties
!    epsi,deli,gami = fixed Thomsen parameters (only where they are not free parameters in the inversion)
!    ranges = maximum and minimum ranges of parameters
!
! Also defined is:
!    np = number of free parameters for given rock physics model
!
! Calls to other routines:
!    Calls functions from the io_lib module to read keywords
!    Stores variables in the insaff_globals module
!
!  Written by J.P. Verdon, University of Bristol, 2008-2015
!***************************************************************************************************

use io_lib 
use insaff_globals
use insaff_lib

implicit none

real,dimension(3)		:: bkg,ae,anu,amu
real,dimension(6,6)		:: c,s
integer					:: unit,lsv
character*256			:: svar

! Get the setup tag
if(.not.get_cmnd_string('setup',settype,lst))then
	settype = 'downhole'
	lst = 8
endif
if (settype(1:lst).eq.'downhole')then 
	isettype = 0
elseif (settype(1:lst).eq.'surface')then
	isettype = 1
else
	stop 'Setup type keyword is not recognised'
endif
write(6,*)'     --> Setup type: ',settype(1:lst)

! Open input file
unit=40
open(unit,file=cfile(1:lc),status='old')

! Get inversion type (mag, psi and combined) and weighting if necessary
if (.not.get_input_string('inversiontype',unit,svar,lsv)) then
	itype=1
else
	select case (svar(1:lsv))
		case ('equal')
			itype=1
			write(6,*)'     --> Inversion type: Equally weighted'
		case ('psi')
			itype=2
			write(6,*)'     --> Inversion type: Psi only'
		case ('dvs')
			itype=3
			write(6,*)'     --> Inversion type: dVs only'
		case ('weighted')
			itype=4
			if (.not.get_input_float('weightfactor',unit,wfact)) wfact=1.0
			write(6,*)'     --> Inversion type: Weighted with W = ',wfact
		case default
			stop 'Inversion type keyword is not recognised'
	end select
endif
if (itype.lt.4)then
	if (get_input_float('weightfactor',unit,wfact))then
		write(6,*)
		write(6,*)'*** Warning - weightfactor has been defined when inversion type is "',&
		svar(1:lsv),'". Weightfactor is ignored'
		wfact=1.0
	endif
endif


! Get rock physics type
if (.not.get_input_string('rockphysicsmodel',unit,svar,lsv)) stop 'Error: Rock Physics Type has not been defined'
select case (svar(1:lsv))
	case ('vti')
		rp=1
	case ('verticalscalar')
		rp=2
	case ('dippingscalar')
		rp=3
	case ('doubleverticalscalar')
		rp=4
	case ('vtiandverticalscalar')
		rp=5
	case ('verticalznzt')
		rp=6
	case ('dippingznzt')
		rp=7
	case ('vtiandverticalznzt')
		rp=8
	case default
		stop 'Rock physics model keyword is not recognised'
end select
write(6,*)'     --> Inverting for RP model type: ',svar(1:lsv)

! Get background properties
if (get_input_mult('background',unit,bkg,3)) then  ! Background velocities and density
	vpi=bkg(1)
	vsi=bkg(2)
	rho=bkg(3)
else
	vpi=4000
	vsi=2000
	rho=2000
endif

! Background Thomsen parameters
if (rp.eq.2.or.rp.eq.3.or.rp.eq.4.or.rp.eq.6.or.rp.eq.7)then
	if (.not.get_input_float('fixedgamma',unit,gami)) gami=0.0
	if (.not.get_input_float('fixeddelta',unit,deli)) deli=0.0
	if (.not.get_input_float('fixedepsilon',unit,epsi))epsi=0.0
else
	if (get_input_float('fixedgamma',unit,gami))then
		write(6,*)
		write(6,*)'*** Warning - fixed gamma has been assigned when it is a free parameter for ',&
		'rock physics model "',svar(1:lsv),'". Fixed gamma is therefore ignored '
		gami=0.0
	endif
	if (get_input_float('fixeddelta',unit,deli))then
		write(6,*)
		write(6,*)'*** Warning - fixed delta has been assigned when it is a free parameter for ',&
		'rock physics model "',svar(1:lsv),'". Fixed delta is therefore ignored '
		deli=0.0
	endif
	if (get_input_float('fixedepsilon',unit,epsi))then
		write(6,*)
		write(6,*)'*** Warning - fixed epsilon has been assigned when it is a free parameter for ',&
		'rock physics model "',svar(1:lsv),'". Fixed epsilon is therefore ignored '
		epsi=0.0
	endif
endif

! Get parameter ranges
select case (rp)
	case (1)	! VTI
		if (.not.get_input_list_mult('ranges','gamma',unit,bkg,2)) stop 'Gamma range has not been defined'
			rng(1,1)=bkg(1)
			rng(2,1)=bkg(2)	
		if (.not.get_input_list_mult('ranges','delta',unit,bkg,2)) stop 'Delta range has not been defined'
			rng(1,2)=bkg(1)
			rng(2,2)=bkg(2)
		if (.not.get_input_list_mult('ranges','epsilon',unit,bkg,2)) stop 'Epsilon range has not been defined'
			rng(1,3)=bkg(1)
			rng(2,3)=bkg(2)
			
	case (2)	! Vertical Fractures
		if (.not.get_input_list_mult('ranges','fracden',unit,bkg,2)) stop 'Fracture density range has not been defined'
			rng(1,1)=bkg(1)
			rng(2,1)=bkg(2)
		if (.not.get_input_list_mult('ranges','strike',unit,bkg,2)) stop 'Fracture strike range has not been defined'
			rng(1,2)=bkg(1)
			rng(2,2)=bkg(2)

	case (3)	! Dipping fractures
		if (.not.get_input_list_mult('ranges','fracden',unit,bkg,2)) stop 'Fracture density range has not been defined'
			rng(1,1)=bkg(1)
			rng(2,1)=bkg(2)
		if (.not.get_input_list_mult('ranges','strike',unit,bkg,2)) stop 'Fracture strike range has not been defined'
			rng(1,2)=bkg(1)
			rng(2,2)=bkg(2)
		if (.not.get_input_list_mult('ranges','dip',unit,bkg,2)) stop 'Fracture dip range has not been defined'
			rng(1,3)=bkg(1)
			rng(2,3)=bkg(2)
			
	case (4)	! Double fracture sets
		if (.not.get_input_list_mult('ranges','fracden1',unit,bkg,2)) stop 'Fracture density 1 range has not been defined'
			rng(1,1)=bkg(1)
			rng(2,1)=bkg(2)
		if (.not.get_input_list_mult('ranges','strike1',unit,bkg,2)) stop 'Fracture strike 1 range has not been defined'
			rng(1,2)=bkg(1)
			rng(2,2)=bkg(2)
		if (.not.get_input_list_mult('ranges','fracden2',unit,bkg,2)) stop 'Fracture density 2 range has not been defined'
			rng(1,3)=bkg(1)
			rng(2,3)=bkg(2)
		if (.not.get_input_list_mult('ranges','strike2',unit,bkg,2)) stop 'Fracture strike 2 range has not been defined'
			rng(1,4)=bkg(1)
			rng(2,4)=bkg(2)
	
	case(5)    ! VTI and scalar fracture
		if (.not.get_input_list_mult('ranges','fracden',unit,bkg,2)) stop 'Fracture density range has not been defined'
			rng(1,1)=bkg(1)
			rng(2,1)=bkg(2)
		if (.not.get_input_list_mult('ranges','strike',unit,bkg,2)) stop 'Fracture strike range has not been defined'
			rng(1,2)=bkg(1)
			rng(2,2)=bkg(2)
		if (.not.get_input_list_mult('ranges','epsilon',unit,bkg,2)) stop 'Epsilon range has not been defined'
			rng(1,3)=bkg(1)
			rng(2,3)=bkg(2)	
		if (.not.get_input_list_mult('ranges','gamma',unit,bkg,2)) stop 'Gamma range has not been defined'
			rng(1,4)=bkg(1)
			rng(2,4)=bkg(2)	
		if (.not.get_input_list_mult('ranges','delta',unit,bkg,2)) stop 'Delta range has not been defined'
			rng(1,5)=bkg(1)
			rng(2,5)=bkg(2)
	
	case (6)	! Vertical Fractures ZnZt
		if (.not.get_input_list_mult('ranges','zt',unit,bkg,2)) stop 'Zn range has not been defined'
			rng(1,1)=bkg(1)
			rng(2,1)=bkg(2)
		if (.not.get_input_list_mult('ranges','znzt',unit,bkg,2)) stop 'Zt range has not been defined'
			rng(1,2)=bkg(1)
			rng(2,2)=bkg(2)
		if (.not.get_input_list_mult('ranges','strike',unit,bkg,2)) stop 'Fracture strike range has not been defined'
			rng(1,3)=bkg(1)
			rng(2,3)=bkg(2)

	case (7)	! Dipping fractures ZnZt
		if (.not.get_input_list_mult('ranges','zt',unit,bkg,2)) stop 'Zn range has not been defined'
			rng(1,1)=bkg(1)
			rng(2,1)=bkg(2)
		if (.not.get_input_list_mult('ranges','znzt',unit,bkg,2)) stop 'Zt range has not been defined'
			rng(1,2)=bkg(1)
			rng(2,2)=bkg(2)
		if (.not.get_input_list_mult('ranges','strike',unit,bkg,2)) stop 'Fracture strike range has not been defined'
			rng(1,3)=bkg(1)
			rng(2,3)=bkg(2)
		if (.not.get_input_list_mult('ranges','dip',unit,bkg,2)) stop 'Fracture dip range has not been defined'
			rng(1,4)=bkg(1)
			rng(2,4)=bkg(2)
	
	case(8)    ! Full VTI and fractures ZnZt
		if (.not.get_input_list_mult('ranges','zt',unit,bkg,2)) stop 'Zn range has not been defined'
			rng(1,1)=bkg(1)
			rng(2,1)=bkg(2)
		if (.not.get_input_list_mult('ranges','znzt',unit,bkg,2)) stop 'Zt range has not been defined'
			rng(1,2)=bkg(1)
			rng(2,2)=bkg(2)
		if (.not.get_input_list_mult('ranges','strike',unit,bkg,2)) stop 'Fracture strike range has not been defined'
			rng(1,3)=bkg(1)
			rng(2,3)=bkg(2)
		if (.not.get_input_list_mult('ranges','epsilon',unit,bkg,2)) stop 'Epsilon range has not been defined'
			rng(1,4)=bkg(1)
			rng(2,4)=bkg(2)	
		if (.not.get_input_list_mult('ranges','gamma',unit,bkg,2)) stop 'Gamma range has not been defined'
			rng(1,5)=bkg(1)
			rng(2,5)=bkg(2)	
		if (.not.get_input_list_mult('ranges','delta',unit,bkg,2)) stop 'Delta range has not been defined'
			rng(1,6)=bkg(1)
			rng(2,6)=bkg(2)
	
end select

! Define number of free parameters
if (rp.eq.1)np=3
if (rp.eq.2)np=2
if (rp.eq.3)np=3
if (rp.eq.4)np=4
if (rp.eq.5)np=5
if (rp.eq.6)np=3
if (rp.eq.7)np=4
if (rp.eq.8)np=6

! Kludge for irevaz
if (.not.get_input_int('irevaz',unit,irevaz))irevaz=0

close(unit)

! Find Gueguen h values
c=cvti(vpi,vsi,rho,0.0,0.0,0.0)
call get_s_nus_mus_and_es(c,s,ae,anu,amu,h)

return
end subroutine input
!***************************************************************************************************


!***************************************************************************************************
subroutine NA_options(nd,nsamplei,nsample,nsleep,ncells,itmax,nclean)
!---------------------------------------------------------------------------------------------------
!
! Subroutine NA_OPTIONS is a heavily edited version based on the original Sambridge call to read in
! the NA tuning parameters. Key changes are that most options are now hardwired. Input file also 
! now takes a keyword-driven format.
!
! Calls to other routines:
!    Calls functions from the io_lib module to read keywords
!    Stores variables in the na_globals module
!    Uses RandomNumbers module to generate a quasi-random seed
!
! Written by J.P. Verdon, University of Bristol, 2008-2011, based on original code by 
! M. Sambridge, RSES, 199?
!***************************************************************************************************

use insaff_globals
use io_lib
use RandomNumbers

implicit none

integer				:: nd,nsamplei,nsample,nsleep,ncells,itmax,nclean
integer				:: nsam,ntotal
integer				:: infolevel,istype,iseed
integer				:: lu_nad,lu_na

common /NA_IO/lu_na,lu_nad   ! na input and nad output file identifiers

! Read maxiteration, initial sample size, subsampling and resampling parameters
if (.not.get_input_list_int('na_options','maxiterations',lu_na,itmax)) stop 'maxiterations not defined'
if (.not.get_input_list_int('na_options','initialsample',lu_na,nsamplei)) stop 'initialsample not defined'
if (.not.get_input_list_int('na_options','subsample',lu_na,nsample)) stop 'subsample not defined'
if (.not.get_input_list_int('na_options','nresample',lu_na,ncells)) stop 'nresample not defined'

! Generate quasi-random seed (is this necessary now that RandomNumbers is used throughout?)
iseed=RN_dateseed()

! Hardwire other parameters
istype=0
nsleep = 1
ntotal = nsamplei + nsample*itmax
nclean = 500
infolevel=0

! perform parameter checking
nsam = max(nsample,nsamplei)

return
end subroutine NA_options
!***************************************************************************************************


	  	
	  	