!***************************************************************************************************
!
!   insaff_lib.f90
!      contains subroutines needed by both insaff and insaff_synthetic, so are best placed in
!      a joint library. Because it contains extrinsic functions, it is set up as a module.
!
!   Written by J.P. Verdon, University of Bristol, 2011.
!
!-------------------------------------------------------------------------------
!  This software is distributed in the hope that it will be useful, but WITHOUT
!  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
!  FITNESS FOR A PARTICULAR PURPOSE.
!
!  This software is distributed for ACADEMIC PURPOSES and is not to be 
!  distributed to any other parties.  This code is NOT TO BE INTEGRATED 
!  within or developed into COMMERCIAL CODE unless WRITTEN PERMISSION 
!  is given by the authors.
!-------------------------------------------------------------------------------
!
!     Last edited: October 2011
!
!***************************************************************************************************

module insaff_lib
contains

!***************************************************************************************************
subroutine read_data(file,l,az,inc,ofast,omag,nmeas,ndec)
!---------------------------------------------------------------------------------------------------
!
! Subroutine READ_DATA reads SWS data from file. 
!   Format of input file should be 4 columns of: azimuth, inclination, fast polarisation, magnitude 
!
! Parameters to be read are:
!    az = S-wave arrival azimuths
!    inc = S-wave arrival inclinations
!    ofast = observed fast direction polarisations
!    omag = observed splitting magnitudes (%Vs)
!
!    subroutine automatically detects the end of the file
!       and returns nmeas = the number of splitting measurements
!
! Calls to other routines: 
!    Uses RandomNumbers module to randomly select data when decimating
!
!  Written by J.P. Verdon, University of Bristol, 2008-2011
!***************************************************************************************************

use RandomNumbers

implicit none

integer,parameter		:: msws=8000
real					:: dummy
real,dimension(msws)	:: az,inc,ofast,omag
integer					:: l,istatus,unit,modulus,i,nmeas,ndec
character(len=*)		:: file

unit=41

! Randomly choose which measurements to take if decimated
modulus=RandomInt(0,ndec-1)

! Open file
open(unit,file=file(1:l),status='old')
nmeas=0  ! Initiate measurement counter
! Loop to read measurements
do i=1,msws
	if (ndec.eq.0.or.mod(i,ndec).eq.modulus)then
		nmeas=nmeas+1
		read(unit,*,iostat=istatus)az(nmeas),inc(nmeas),ofast(nmeas),omag(nmeas)
   		if (istatus.lt.0) exit
   		
   		! Set az to between 0 - 360 (not sure this is strictly necessary, but hey, I'm a neat freak...)
   		az(nmeas) = modulo(az(nmeas),360.0)

   		!inc(nmeas)=180-inc(nmeas)
   	else
		read(unit,*,IOSTAT=istatus)dummy
		if (istatus.lt.0)exit
	endif	
enddo
nmeas=nmeas-1

close(unit)

return
end subroutine read_data
!***************************************************************************************************


!***************************************************************************************************
subroutine read_angles(file,l,az,inc,nmeas,ndec)
!---------------------------------------------------------------------------------------------------
!
! Subroutine READ_ANGLES reads SWS arrival angles data from file. 
!   Format of input file should be 2 columns of: azimuth, inclination
!
! Parameters to be read are:
!    az = S-wave arrival azimuths
!    inc = S-wave arrival inclinations
!
!    subroutine automatically detects the end of the file
!       and returns nmeas = the number of splitting measurements
!
! Calls to other routines: 
!    Uses RandomNumbers module to randomly select data when decimating
!
!  Written by J.P. Verdon, University of Bristol, 2008-2011
!***************************************************************************************************

use RandomNumbers

implicit none

integer,parameter		:: msws=8000
real					:: dummy
real,dimension(msws)	:: az,inc
integer					:: l,istatus,unit,modulus,i,nmeas,ndec
character*256			:: file

unit=41

! Randomly choose which measurements to take if decimated
modulus=RandomInt(0,ndec-1)

! Open file
open(unit,file=file(1:l),status='old')
nmeas=0  ! Initiate measurement counter
! Loop to read measurements
do i=1,msws
	if (ndec.eq.0.or.mod(i,ndec).eq.modulus)then
		nmeas=nmeas+1
		read(unit,*,iostat=istatus)az(nmeas),inc(nmeas)
   		if (istatus.lt.0) exit
		if (az(nmeas).gt.360)az(nmeas)=az(nmeas)-360
   		!inc(nmeas)=180-inc(nmeas)
   		!write(*,*)az(nmeas),inc(nmeas)
   		!read(*,*)
   	else
		read(unit,*,IOSTAT=istatus)dummy
		if (istatus.lt.0)exit
	endif	
enddo
nmeas=nmeas-1

close(unit)

return
end subroutine read_angles
!***************************************************************************************************


!***************************************************************************************************
subroutine write_cijkl(c,rho)
!---------------------------------------------------------------------------------------------------
!
! Subroutine WRITE_CIJKL write the best fit stiffness tensor to file, in both ATRAK and 6x6 format
!
! Call no other subroutines
!
!***************************************************************************************************

implicit none

integer						:: i,j
real,dimension(6,6)			:: c
real						:: rho

! 6x6 format
open(18,file='./Cijkl.dat',form='formatted')
do i=1,6
	write(18,95)(c(i,j),j=1,6)
enddo
close(18)

! Atrak format (density normalised)
open(21,file='./C_atrak.dat',form='formatted')
do i=1,6
	do j=i,6
    	write(21,92)i,' ',j,' ',c(i,j)/rho
    enddo
enddo
write(21,91)'7 7 ',rho
close(21)

92    format(i1,a1,i1,a1,f11.1)
91    format(a4,f11.1)
95    format(6(EN14.3))

return
end subroutine write_cijkl
!***************************************************************************************************


!***************************************************************************************************
subroutine write_syn_summary(unit,rp,irp,model,ranges,ivp,ivs,irho,ixi,ixi2,ialp,ialp2,igam,idel,ieps,&
	idip,vpi,vsi,gami,epsi,deli,nafile,lnaf,sfile,ls)
!---------------------------------------------------------------------------------------------------
!
! Subroutine WRITE_NA_SUMMARY writes summary information from the insaff_na inversion
!
! Information written includes:
!    vpi,vsi,epsi,gami,deli = initial, fixed parameters
!    ranges = maximum and minimum parameter ranges
!    model = best fit model
!    lb = axis labels for plotting
!
!    Information is written to file identified by unit
!
! Written by J.P. Verdon, University of Bristol, 2008-2011
!
!***************************************************************************************************

implicit none

integer					:: irp,rp,unit,lnaf,ls
real					:: model(*),ranges(2,*)
real					:: ivp,ivs,irho,ixi,ixi2,ialp,ialp2,igam,idel,ieps,idip
real					:: vpi,vsi,gami,epsi,deli
character*256			:: nafile,sfile

write(unit,101)'in1=',ivp
write(unit,101)'in2=',ivs
write(unit,100)'in3=',epsi
write(unit,100)'in4=',gami
write(unit,100)'in5=',deli
write(unit,100)'or1=',ranges(1,1)
write(unit,100)'or2=',ranges(1,2)
write(unit,100)'or3=',ranges(1,3)
write(unit,100)'or4=',ranges(1,4)
write(unit,100)'or5=',ranges(1,5)
write(unit,100)'or6=',ranges(1,6)
write(unit,100)'en1=',ranges(2,1)
write(unit,100)'en2=',ranges(2,2)
write(unit,100)'en3=',ranges(2,3)
write(unit,100)'en4=',ranges(2,4)
write(unit,100)'en5=',ranges(2,5)
write(unit,100)'en6=',ranges(2,6)
select case (rp)
	case (1)
		write(unit,100)'bf1=',model(1)
		write(unit,100)'bf2=',model(2)
		write(unit,100)'bf3=',model(3)
		write(unit,100)'bf4=',ranges(1,4)
		write(unit,100)'bf5=',ranges(1,5)
		write(unit,100)'bf6=',ranges(1,6)
		write(unit,'(A9)')'lb1=Gamma'
		write(unit,'(A9)')'lb2=Delta'
		write(unit,'(A11)')'lb3=Epsilon'
	case (2)
		write(unit,100)'bf1=',model(1)
		write(unit,100)'bf2=',model(2)
		write(unit,100)'bf3=',ranges(1,3)
		write(unit,100)'bf4=',ranges(1,4)
		write(unit,100)'bf5=',ranges(1,5)
		write(unit,100)'bf6=',ranges(1,6)
		write(unit,'(A19)')'lb1=FractureDensity'
		write(unit,'(A18)')'lb2=FractureStrike'
	case (3)
		write(unit,100)'bf1=',model(1)
		write(unit,100)'bf2=',model(2)
		write(unit,100)'bf3=',model(3)
		write(unit,100)'bf4=',ranges(1,4)
		write(unit,100)'bf5=',ranges(1,5)
		write(unit,100)'bf6=',ranges(1,6)
		write(unit,'(A19)')'lb1=FractureDensity'
		write(unit,'(A18)')'lb2=FractureStrike'
		write(unit,'(A15)')'lb3=FractureDip'
	case (4)
		write(unit,100)'b1=',model(1)
		write(unit,100)'b2=',model(2)
		write(unit,100)'b3=',model(3)
		write(unit,100)'b4=',model(4)
		write(unit,100)'b5=',ranges(1,5)
		write(unit,100)'bf6=',ranges(1,6)
		write(unit,'(A19)')'l1=FractureDensity'
		write(unit,'(A18)')'l2=FractureStrike'
		write(unit,'(A9)')'l3=Gamma'
		write(unit,'(A9)')'l4=Delta'
	case (5)
		write(unit,100)'bf1=',model(1)
		write(unit,100)'bf2=',model(2)
		write(unit,100)'bf3=',model(3)
		write(unit,100)'bf4=',model(4)
		write(unit,100)'bf5=',ranges(1,5)
		write(unit,100)'bf6=',ranges(1,6)
		write(unit,'(A20)')'lb1=FractureDensity1'
		write(unit,'(A19)')'lb2=FractureStrike1'
		write(unit,'(A20)')'lb3=FractureDensity2'
		write(unit,'(A19)')'lb4=FractureStrike2'
	case (6)
		write(unit,100)'bf1=',model(1)
		write(unit,100)'bf2=',model(2)
		write(unit,100)'bf3=',model(3)
		write(unit,100)'bf4=',model(4)
		write(unit,100)'bf5=',model(5)
		write(unit,100)'bf6=',ranges(1,6)
		write(unit,'(A19)')'lb1=FractureDensity'
		write(unit,'(A18)')'lb2=FractureStrike'
		write(unit,'(A11)')'lb3=Epsilon'
		write(unit,'(A9)')'lb4=Gamma'
		write(unit,'(A9)')'lb5=Delta'
	case (7)
		write(unit,100)'bf1=',model(1)
		write(unit,100)'bf2=',model(2)
		write(unit,100)'bf3=',model(3)
		write(unit,100)'bf4=',ranges(1,4)
		write(unit,100)'bf5=',ranges(1,5)
		write(unit,100)'bf6=',ranges(1,6)
		write(unit,'(A6)')'lb1=Zn'
		write(unit,'(A6)')'lb2=Zt'
		write(unit,'(A18)')'lb3=FractureStrike'
	case (8)	
		write(unit,100)'bf1=',model(1)
		write(unit,100)'bf2=',model(2)
		write(unit,100)'bf3=',model(3)
		write(unit,100)'bf4=',model(4)
		write(unit,100)'bf5=',ranges(1,5)
		write(unit,100)'bf6=',ranges(1,6)
		write(unit,'(A6)')'lb1=Zn'
		write(unit,'(A6)')'lb2=Zt'
		write(unit,'(A18)')'lb3=FractureStrike'
		write(unit,'(A15)')'lb4=FractureDip'
	case (9)
		write(unit,100)'bf1=',model(1)
		write(unit,100)'bf2=',model(2)
		write(unit,100)'bf3=',model(3)
		write(unit,100)'bf4=',model(4)
		write(unit,100)'bf5=',model(5)
		write(unit,100)'bf6=',model(6)
		write(unit,'(A6)')'lb1=Zn'
		write(unit,'(A6)')'lb1=ZnZt'
		write(unit,'(A18)')'lb3=FractureStrike'
		write(unit,'(A11)')'lb4=Epsilon'
		write(unit,'(A9)')'lb5=Gamma'
		write(unit,'(A9)')'lb6=Delta'
end select

write(unit,*)'sws='//sfile(1:ls)
if (lnaf.gt.0)then
	write(unit,*)'nad='//nafile(1:lnaf)
endif

100 format(A4,F7.3)
101 format(A4,F7.1)

return
end subroutine write_syn_summary
!***************************************************************************************************



!***************************************************************************************************
subroutine write_cube(unit,mfnorm,ngrid)
!***************************************************************************************************
implicit none

integer					:: unit,ngrid,i
real,dimension(ngrid)	:: mfnorm

do i=1,ngrid
	write(unit)mfnorm(i)
enddo

return
end subroutine write_cube
!***************************************************************************************************


!***************************************************************************************************
subroutine write_gs_summary(unit,model,rp,steps,ranges,err,vpi,vsi,gami,epsi,deli)
!***************************************************************************************************

implicit none

integer					:: rp,steps(*),unit
real					:: xif,xi2f,alpf,gamf,delf,epsf,dipf,ranges(2,*),err(*),model(*)
real					:: vpi,vsi,gami,epsi,deli

write(unit,101)'in1=',vpi
write(unit,101)'in2=',vsi
write(unit,100)'in3=',epsi
write(unit,100)'in4=',gami
write(unit,100)'in5=',deli
write(unit,'(A4,I2)')'ns1=',steps(1)
write(unit,'(A4,I2)')'ns2=',steps(2)
write(unit,'(A4,I2)')'ns3=',steps(3)
write(unit,'(A4,I2)')'ns4=',steps(4)
write(unit,'(A4,I2)')'ns5=',steps(5)
write(unit,100)'or1=',ranges(1,1)
write(unit,100)'or2=',ranges(1,2)
write(unit,100)'or3=',ranges(1,3)
write(unit,100)'or4=',ranges(1,4)
write(unit,100)'or5=',ranges(1,5)
write(unit,100)'en1=',ranges(2,1)
write(unit,100)'en2=',ranges(2,2)
write(unit,100)'en3=',ranges(2,3)
write(unit,100)'en4=',ranges(2,4)
write(unit,100)'en5=',ranges(2,5)
write(unit,100)'mf1=',err(1)
write(unit,100)'mf2=',err(2)
write(unit,100)'mf3=',err(3)
write(unit,100)'mf4=',err(4)
write(unit,100)'mf5=',err(5)
write(unit,100)'bf1=',model(1)
write(unit,100)'bf2=',model(2)
write(unit,100)'bf3=',model(3)
write(unit,100)'bf4=',model(4)
write(unit,100)'bf5=',model(5)
select case (rp)
	case (1)
		write(unit,'(A9)')'lb1=Gamma'
		write(unit,'(A9)')'lb2=Delta'
		write(unit,'(A11)')'lb3=Epsilon'
	case (2)
		write(unit,'(A19)')'lb1=FractureDensity'
		write(unit,'(A18)')'lb2=FractureStrike'
	case (3)
		write(unit,'(A19)')'lb1=FractureDensity'
		write(unit,'(A18)')'lb2=FractureStrike'
		write(unit,'(A15)')'lb3=FractureDip'
	case (4)
		write(unit,'(A19)')'lb1=FractureDensity'
		write(unit,'(A18)')'lb2=FractureStrike'
		write(unit,'(A9)')'lb3=Gamma'
		write(unit,'(A9)')'lb4=Delta'
	case (5)
		write(unit,'(A20)')'lb1=FractureDensity1'
		write(unit,'(A19)')'lb2=FractureStrike1'
		write(unit,'(A20)')'lb3=FractureDensity2'
		write(unit,'(A19)')'lb4=FractureStrike2'
	case (6)
		write(unit,'(A19)')'lb1=FractureDensity'
		write(unit,'(A18)')'lb2=FractureStrike'
		write(unit,'(A11)')'lb3=Epsilon'
		write(unit,'(A9)')'lb4=Gamma'
		write(unit,'(A9)')'lb5=Delta'
end select

100 format(A4,F7.3)
101 format(A4,F7.1)

return
end subroutine write_gs_summary
!***************************************************************************************************


!***************************************************************************************************
subroutine write_vectors(c,rho)
!---------------------------------------------------------------------------------------------------
!
! Subroutine WRITE_VECTORS writes to file the modelled splitting operators for all angles of
! arrival. These are used by hemiplot.gmt to generate a hemisphere projection of modelled splitting
!
! Calls to subroutines:
!		calls anisvelocity to compute the splitting operators from a stiffness tensor
!
! Two files are created by write_vectors:
!       ./phase.dat - writes magnitude and fast direction at all points in loop over arrival angles.
!                     This is used for contour plotting    
!       ./vectors.dat - only writes information at specified steps. This is used to plot the ticks
!
! Written by J.P. Verdon, University of Bristol, 2008-2011
!
!***************************************************************************************************

implicit none

integer					:: i,j,istep,iinc,iaz
real,dimension(6,6)		:: c
real					:: rho,fast,mag,az,inc,vp,vsf,vss

open(19,file='./phase.dat')
open(20,file='./vectors.dat')	  

! Loop over inclinations 
do iinc=0,90,10
	inc=real(iinc)
	istep=1111
	! Set up steps for selected vectors
	if (iinc.ge.10)istep=90
	if (iinc.ge.20)istep=60
	if (iinc.ge.30)istep=45
	if (iinc.ge.40)istep=40
	if (iinc.ge.50)istep=18
	if (iinc.ge.70)istep=18
	if (iinc.ge.80)istep=18
	if (iinc.ge.90)istep=12

	! Loop over azimuths	
	do iaz=0,360,1
		az=real(iaz)
		! Compute splitting operators
		call anisvelocity(c,rho,az,90-inc,vp,vsf,vss,fast)
		mag=100*(vsf-vss)/((vsf+vss)/2)
		
		! Write magnitude at every arrival node
		write(19,94),az,inc,mag
		
		! Write fast direction and magnitude at selected arrival nodes
		if (mod(iaz,istep).eq.0.and.iaz.ne.0) then
			write(20,93), az,inc,fast,mag
		endif
     
	enddo 
enddo
	  
close(19)
close(20)
	  
94      format(2(F8.1), 6(F10.4))
93      format(2(F8.1), 2(F10.4))

return
end subroutine write_vectors
!***************************************************************************************************


!***************************************************************************************************
subroutine write_to_screen(model,rp,error)
!---------------------------------------------------------------------------------------------------
!
! Subroutine WRITE_TO_SCREEN writes the best fit model parameters to standard output
!
! Call no other subroutines
!
!***************************************************************************************************

implicit none

integer					:: rp
real					:: model(*)
real,optional			:: error(*)

if (present(error))then
	select case (rp)
		case (1)
			write(6,98)'Best fit gamma: ',model(1),' ± ',error(1) 
			write(6,98)'Best fit delta: ',model(2),' ± ',error(2)
			write(6,98)'Best fit epsilon: ',model(3),' ± ',error(3)
		case (2)
			write(6,98)'Best fit fracture density: ',model(1),' ± ',error(1)
			write(6,98)'Best fit fracture strike: ',model(2),' ± ',error(2)
		case (3)
			write(6,98)'Best fit fracture density: ',model(1),' ± ',error(1)
			write(6,98)'Best fit fracture strike: ',model(2),' ± ',error(2)
			write(6,98)'Best fit fracture dip: ',model(3),' ± ',error(3)
		case (4)
			write(6,98)'Best fit fracture density: ',model(1),' ± ',error(1)
			write(6,98)'Best fit fracture strike: ',model(2),' ± ',error(2)
			write(6,98)'Best fit gamma: ',model(3),' ± ',error(3)
			write(6,98)'Best fit delta: ',model(4),' ± ',error(4)
		case (5)
			write(6,98)'Best fit fracture density 1: ',model(1),' ± ',error(1)
			write(6,98)'Best fit fracture strike 1: ',model(2),' ± ',error(2)
			write(6,98)'Best fit fracture density 2: ',model(3),' ± ',error(3)
			write(6,98)'Best fit fracture strike 2: ',model(4),' ± ',error(4)
		case (6)
			write(6,98)'Best fit fracture density: ',model(1),' ± ',error(1)
			write(6,98)'Best fit fracture strike: ',model(2),' ± ',error(2)
			write(6,98)'Best fit epsilon: ',model(3),' ± ',error(3)
			write(6,98)'Best fit gamma: ',model(4),' ± ',error(4)
			write(6,98)'Best fit delta: ',model(5),' ± ',error(5)
	end select
else
	select case (rp)
		case (1)
			write(6,98)'Best fit gamma: ',model(1)
			write(6,98)'Best fit delta: ',model(2)
			write(6,98)'Best fit epsilon: ',model(3)
		case (2)
			write(6,98)'Best fit fracture density: ',model(1)
			write(6,98)'Best fit fracture strike: ',model(2)
		case (3)
			write(6,98)'Best fit fracture density: ',model(1)
			write(6,98)'Best fit fracture strike: ',model(2)
			write(6,98)'Best fit fracture dip: ',model(3)
		case (4)
			write(6,98)'Best fit fracture density 1: ',model(1)
			write(6,98)'Best fit fracture strike 1: ',model(2)
			write(6,98)'Best fit fracture density 2: ',model(3)
			write(6,98)'Best fit fracture strike 2: ',model(4)
		case (5)
			write(6,98)'Best fit fracture density: ',model(1)
			write(6,98)'Best fit fracture strike: ',model(2)
			write(6,98)'Best fit epsilon: ',model(3)
			write(6,98)'Best fit gamma: ',model(4)
			write(6,98)'Best fit delta: ',model(5)
		case (6)
			write(6,98)'Best fit Zn: ',model(1)
			write(6,98)'Best fit Zn/Zt: ',model(2)
			write(6,98)'Best fit fracture strike: ',model(3)
		case (7)
			write(6,98)'Best fit Zn: ',model(1)
			write(6,98)'Best fit Zn/Zt: ',model(2)
			write(6,98)'Best fit fracture strike: ',model(3)
			write(6,98)'Best fit fracture dip: ',model(4)
		case (8)
			write(6,98)'Best fit Zt: ',model(1)
			write(6,98)'Best fit Zn/Zt: ',model(2)
			write(6,98)'Best fit fracture strike: ',model(3)
			write(6,98)'Best fit epsilon: ',model(4)
			write(6,98)'Best fit gamma: ',model(5)
			write(6,98)'Best fit delta: ',model(6)
	end select
endif

98       format(A,f9.3,A3,f9.4)
return
end subroutine write_to_screen
!***************************************************************************************************

! From here - functions for generating stiffness tensors based on RP models
!***************************************************************************************************
real function cvti(vp,vs,rho,eps,gam,del)
!---------------------------------------------------------------------------------------------------
!
! Function CVTI generates a vti stiffness tensor based on input P and S velocities and
! Thomsen's parameters
!
! Input parameters are:
!    vp = P-wave velocity
!    vs = S-wave velocity
!    eps = Thomsen's epsilon
!    gam = Thomsen's gamma
!    del = Thomsen's delta
!
! Calls to other subroutines:
!    zeros2d - initialise matrix full of zeros
!	 symmetry2d - to fill matrix
!
! Written by J.P. Verdon, University of Bristol, 2008-2011
!
!***************************************************************************************************

implicit none

real			:: vp,vs,rho,eps,gam,del
dimension		:: cvti(6,6)

call zeros2d(cvti,6,6)
cvti(3,3)=vp*vp*rho
cvti(4,4)=vs*vs*rho
cvti(1,1)=cvti(3,3)*(2*eps+1)
cvti(6,6)=cvti(4,4)*(2*gam+1)	  
cvti(2,2)=cvti(1,1)
cvti(5,5)=cvti(4,4)
cvti(1,2)=cvti(1,1)-2*cvti(6,6)	
cvti(1,3)=sqrt(2*del*cvti(3,3)*(cvti(3,3)-cvti(4,4)) + (cvti(3,3)-cvti(4,4))*(cvti(3,3)-cvti(4,4))) - cvti(4,4)     
if (2*del*cvti(3,3)*(cvti(3,3)-cvti(4,4)) + (cvti(3,3)-cvti(4,4))*(cvti(3,3)-cvti(4,4)).lt.0)then
	write(*,*)'stopping program, c(1,3) is not viable'
	read(*,*)
	stop
endif
cvti(2,3)=cvti(1,3)
call symmetry2d(cvti,6)      

return
end function cvti
!***************************************************************************************************


!***************************************************************************************************
real function cdfrac(vp,vs,rho,xi,alp,xi2,alp2,eps,gam,del)
!---------------------------------------------------------------------------------------------------
!
! Function CDFRAC generates a stiffness tensor for two vertical fracture sets and a vti fabric
!
! Input parameters are:
!    vp = P-wave velocity
!    vs = S-wave velocity
!    eps = Thomsen's epsilon
!    gam = Thomsen's gamma
!    del = Thomsen's delta
!    xi,xi2 = Fracture densities of the 2 fracture sets
!    alp,alp2 = strikes of the 2 fracture sets
!
! Calls to other subroutines:
!    calc_b - compute Bn and Bt based on Lame parameters
!    cvti - generate a VTI tensor
!    invmat - invert between compliance/stiffness
!    zeros2d - initialise tensors
!    a_ijkl - convert from Voigt notation to 3x3x3x3 in stiffness domain
!    a_ij - convert from 3x3x3x3 to Voigt notation in stiffness domain
!    rotatelc - elastic tensor rotation
!
! Written by J.P. Verdon, University of Bristol, 2008-2011
!
!***************************************************************************************************

implicit none

real						:: vp,vs,rho,xi,xi2,alp,alp2,eps,gam,del,mu,lam,bn,bt1,bt2
real						:: alpha,alpha2,alphadif,lamv,muv
real,dimension(6,6)			:: c,s
real,dimension(3,3)			:: rot
real,dimension(3,3,3,3)		:: crot3,cout3
real,parameter				:: pi=3.141592654
dimension					:: cdfrac(6,6)

! Put alphas into Mainprice coords
alpha=90-alp
alpha2=90-alp2
alphadif=alpha-alpha2

c=cvti(vp,vs,rho,eps,gam,del)   ! Construct the VTI background

! FIRST FRACTURE SET
! Compute crack normal and tangential compliances
mu=c(6,6)
lam=c(2,2)-2.0*mu
bn = calc_b(1,lam,mu,xi)
bt1 = calc_b(2,lam,mu,xi) ! bt1 is the vertical
muv=c(4,4)
lamv=c(3,3)-2.0*mu
bt2 = calc_b(2,lamv,muv,xi)  ! bt2 is the vertical

! Add compliance to background
call invmat(c,6,s)  
s(1,1)=s(1,1)+bn
s(5,5)=s(5,5)+bt2
s(6,6)=s(6,6)+bt1
! Invert for stiffness matrix:	  
call invmat(s,6,c)

! Construct the 3x3 rotation matrix	
call zeros2d(rot,3,3)
rot(3,3)=1.0
rot(1,1)=cos(alphadif*pi/180)
rot(2,2)=cos(alphadif*pi/180)
rot(1,2)=0.0-sin(alphadif*pi/180)
rot(2,1)=0.0-rot(1,2)

! Rotate to strike
call a_ijkl(c,crot3)
call rotatelc(rot,crot3,cout3)
call a_ij(cout3,c)

! SECOND FRACTURE SET
! Compute normal and tangential compliances
bn=calc_b(1,lam,mu,xi2)
bt1=calc_b(2,lam,mu,xi2)  !bt1 is the horizontal
bt2=calc_b(2,lamv,muv,xi2) !bt2 is vertical

! Add the 2nd fractures to the VTI+fracs compliance
call invmat(c,6,s)
s(1,1)=s(1,1)+bn
s(5,5)=s(5,5)+bt2
s(6,6)=s(6,6)+bt1
	  
! Invert for stiffness matrix:	  
call invmat(s,6,c)

! Construct the 3x3 rotation matrix	
call zeros2d(rot,3,3)
rot(3,3)=1.0
rot(1,1)=cos(alpha2*pi/180)
rot(2,2)=cos(alpha2*pi/180)
rot(1,2)=0.0-sin(alpha2*pi/180)
rot(2,1)=0.0-rot(1,2)

! Rotate to strike
call a_ijkl(c,crot3)
call rotatelc(rot,crot3,cout3)
call a_ij(cout3,cdfrac)

return
end function cdfrac
!***************************************************************************************************


!***************************************************************************************************
real function cvtifrac(vp,vs,rho,xi,alp,eps,gam,del)
!---------------------------------------------------------------------------------------------------
!
! Function CVTIFRAC generates a stiffness tensor for one vertical fracture set and a vti fabric
!
! Input parameters are:
!    vp = P-wave velocity
!    vs = S-wave velocity
!    eps = Thomsen's epsilon
!    gam = Thomsen's gamma
!    del = Thomsen's delta
!    xi = Fracture density of the fracture set
!    alp = strike of the fracture set
!
! Calls to other subroutines:
!    calc_b - compute Bn and Bt based on Lame parameters
!    cvti - generate a VTI tensor
!    invmat - invert between compliance/stiffness
!    zeros2d - initialise tensors
!    a_ijkl - convert from Voigt notation to 3x3x3x3 in stiffness domain
!    a_ij - convert from 3x3x3x3 to Voigt notation in stiffness domain
!    rotatelc - elastic tensor rotation
!
! Written by J.P. Verdon, University of Bristol, 2008-2011
!
!***************************************************************************************************

implicit none
integer						::i,j
real						:: vp,vs,rho,xi,alp,eps,gam,del,mu,lam,rot(3,3),alpha,bn,bt1,bt2
real,dimension(6,6)			:: c,s
real,dimension(3,3,3,3)		:: crot3,cout3
real,parameter				:: pi=3.1415926535897932
dimension					:: cvtifrac(6,6)

alpha=90-alp	! Put strike into Mainprice coords

c=cvti(vp,vs,rho,eps,gam,del)   ! Construct the VTI background

! Compute fracture compliances	  
mu=c(6,6)
lam=c(2,2)-2.0*mu
bn = calc_b(1,lam,mu,xi)
bt1 = calc_b(2,lam,mu,xi) ! bt1 is the vertical

mu=c(4,4)
lam=c(3,3)-2.0*mu
bt2 = calc_b(2,lam,mu,xi)  ! bt2 is the vertical

! Add compliance to background
call invmat(c,6,s)  
s(1,1)=s(1,1)+bn
s(5,5)=s(5,5)+bt2
s(6,6)=s(6,6)+bt1
! Invert for stiffness matrix:	  
call invmat(s,6,c)

! Construct the 3x3 rotation matrix	
call zeros2d(rot,3,3)
rot(3,3)=1.0
rot(1,1)=cos(alpha*pi/180)
rot(2,2)=cos(alpha*pi/180)
rot(1,2)=0.0-sin(alpha*pi/180)
rot(2,1)=0.0-rot(1,2)

! Rotate to strike
call a_ijkl(c,crot3)
call rotatelc(rot,crot3,cout3)
call a_ij(cout3,cvtifrac)

return
end function cvtifrac
!***************************************************************************************************


!***************************************************************************************************
real function cvtifrac_znzt(vp,vs,rho,znh,zth,alp,eps,gam,del)
!---------------------------------------------------------------------------------------------------
!
! Function CVTIFRAC_ZNZT generates a stiffness tensor for one vertical fracture set and a vti fabric
!  using an input zn and zt (rather than fracture density)
!
! Input parameters are:
!    vp = P-wave velocity
!    vs = S-wave velocity
!    eps = Thomsen's epsilon
!    gam = Thomsen's gamma
!    del = Thomsen's delta
!    xi = Fracture density of the fracture set
!    alp = strike of the fracture set
!
! Calls to other subroutines:
!    cvti - generate a VTI tensor
!    invmat - invert between compliance/stiffness
!    zeros2d - initialise tensors
!    a_ijkl - convert from Voigt notation to 3x3x3x3 in stiffness domain
!    a_ij - convert from 3x3x3x3 to Voigt notation in stiffness domain
!    rotatelc - elastic tensor rotation
!
! Written by J.P. Verdon, University of Bristol, 2008-2011
!
!***************************************************************************************************

implicit none
integer						:: i,j
real						:: vp,vs,rho,alp,eps,gam,del,mu,lam,rot(3,3),alpha
real						:: znh,zth,zn,zt
real,dimension(6,6)			:: c,s
real,dimension(3,3,3,3)		:: crot3,cout3
real,dimension(3)			:: h,ae,anu,amu
real,parameter				:: pi=3.1415926535897932
dimension					:: cvtifrac_znzt(6,6)

alpha=90-alp	! Put strike into Mainprice coords

! Find Gueguen h values
c=cvti(vp,vs,rho,0.0,0.0,0.0)
call get_s_nus_mus_and_es(c,s,ae,anu,amu,h)

zn=znh/h(1)
zt=zth/h(1)

c=cvti(vp,vs,rho,eps,gam,del)   ! Construct the VTI background

! Add compliance to background
call invmat(c,6,s)  
s(1,1)=s(1,1)+zn
s(5,5)=s(5,5)+zt
s(6,6)=s(6,6)+zt
! Invert for stiffness matrix:	  
call invmat(s,6,c)

! Construct the 3x3 rotation matrix	
call zeros2d(rot,3,3)
rot(3,3)=1.0
rot(1,1)=cos(alpha*pi/180)
rot(2,2)=cos(alpha*pi/180)
rot(1,2)=0.0-sin(alpha*pi/180)
rot(2,1)=0.0-rot(1,2)

! Rotate to strike
call a_ijkl(c,crot3)
call rotatelc(rot,crot3,cout3)
call a_ij(cout3,cvtifrac_znzt)

return
end function cvtifrac_znzt
!***************************************************************************************************

!***************************************************************************************************
real function cdipfr(vp,vs,rho,xi,alp,dip,eps,gam,del)
!---------------------------------------------------------------------------------------------------
!
! Function CDIPFR generates a stiffness tensor for one dipping fracture set and a vti fabric
!
! Input parameters are:
!    vp = P-wave velocity
!    vs = S-wave velocity
!    eps = Thomsen's epsilon
!    gam = Thomsen's gamma
!    del = Thomsen's delta
!    xi = Fracture density of the fracture set
!    alp = strike of the fracture set
!    dip = dip of the fracture set
!
! Calls to other subroutines:
!    calc_b - compute Bn and Bt based on Lame parameters
!    cvti - generate a VTI tensor
!    krondel - generate the Kroenecker delta
!    invmat - invert between compliance/stiffness
!    zeros4d - initialise 4D tensors
!    s_ij - convert from 3x3x3x3 to Voigt notation in compliance domain
!
! Written by J.P. Verdon, University of Bristol, 2008-2011
!
!***************************************************************************************************

implicit none

integer					:: i,j,k,l
real					:: vp,vs,rho,xi,alp,dip,eps,gam,del,alpha,beta,mu,lam,bn,bt,bt1,bt2
real					:: x(3)
real,dimension(3,3)		:: akr,z
real,dimension(6,6)		:: c,sf,s,sb
real,dimension(3,3,3,3)	:: ssft,ssf
real,parameter			:: pi=3.141592654
dimension				:: cdipfr(6,6)

! Put alpha into Mainprice coords
alpha=90-alp	 
beta=dip
alpha=alpha*pi/180
beta=beta*pi/180

! Construct the VTI background
c=cvti(vp,vs,rho,eps,gam,del)   ! Construct the VTI background

! Compute fracture compliances	  
mu=c(6,6)
lam=c(1,1)-2.0*mu
bn = calc_b(1,lam,mu,xi)
bt1 = calc_b(2,lam,mu,xi) ! bt1 is the vertical
mu=c(4,4)
lam=c(3,3)-2.0*mu
bt2 = calc_b(2,lam,mu,xi)  ! bt2 is the vertical
bt=bt2 !(bt1+bt2)/2   ! Use the average bt
      
! Add crack compliance to compliance tensor
x(1) = cos(beta)*cos(alpha)
x(2) = cos(beta)*sin(alpha)
x(3) = sin(beta)
do i=1,3
	if (abs(x(i)).lt.0.001)x(i) = 0.0
enddo
	  
call krondel(akr,3)     
do i = 1,3
	do j = 1,3
    	z(i,j) = bt*akr(i,j) + (bn-bt)*x(i)*x(j)
    enddo
enddo

! Calculation of additional compliances due to single set of aligned
! fractures using equation (6) of Schoenberg and Sayers (1995)
call zeros4d(ssft,3,3,3,3)	  

do i=1,3
	do j=1,3
    	do k=1,3
        	do l=1,3
            	ssf(i,j,k,l)= z(i,k)*x(l)*x(j)+z(j,k)*x(l)*x(i)+z(i,l)*x(k)*x(j)+z(j,l)*x(k)*x(i)
                ssft(i,j,k,l) = ssft(i,j,k,l) + 0.25*ssf(i,j,k,l)
            enddo
         enddo
	enddo
enddo

call s_ij(ssft,sf)
call invmat(c,6,sb)
s=sb+sf
call invmat(s,6,cdipfr)
	  
return
end function cdipfr
!***************************************************************************************************


!***************************************************************************************************
function calc_b(itype,lam,mu,xi)
!---------------------------------------------------------------------------------------------------
!
! Function CALC_B computes the normal and tangential stiffness tensors as a function of Lame 
! parameters
!
! Input parameters are:
!     itype - defines whether to return normal or tangential compliance
!     lam,mu - Lame parameters
!     xi - fracture density
!
! Calls no other subroutines
!
! Written by J.P. Verdon, University of Bristol, 2008-2011
!
!***************************************************************************************************

implicit none

integer			:: itype
real			:: lam,mu,xi,calc_b
	  
if (itype.eq.1)then ! Crack normal
	calc_b=4*xi*(lam+2*mu)/(3*mu*(lam+mu))
elseif (itype.eq.2)then  ! Crack tangential
	calc_b=16*xi*(lam+2*mu)/(3*mu*(3*lam+4*mu))
endif
	  
return
end function calc_b
	  
!***************************************************************************************************

end module insaff_lib
!***************************************************************************************************

