function [asp,xi,znzt,alpha,vin,vbc,vt,xibc,quality]=vakh(vdata,cdata,method,figures)
%
% function [asp,xi,znzt,alpha,vin,vbc,vt,xibc,quality]=vakh(vdata,cdata,method,figures)
%
% VAKH (Verdon, Angus, Kendall and Hall) method to compute microcrack
% densities and aspect ratios, and Zn/Zt, based on triaxial ultrasonic
% velocity measurements.
%
% See Verdon et al., (2008, Geophysics) for more details
%
% INPUTS:   vin =   Input velocities. Can be either a string to read a file 
%                   or a matrix of values. Can have either 2 values (Vp and
%                   Vs) for isotropic, or 7 values (Vpx, Vpy, Vpz, Vp45, 
%                   Vsyz, Vsxz, Vsxy)                     
%           cdata =  Stiffness tensor for the high pressure asymptote. Can
%                   be a string to read a file, a set of values (6x6 tensor)
%                   or 'high' to use the maximum velocity
%           method = either 'scalar' or 'beta' to use scalar cracks or to
%                   include Zn/Zt
%           figures = tag to plot all figures ("all"), some figures
%                   ("basic") or no figures ("none")
%
% OUTPUTS:  asp =   crack initial aspect ratios
%           xi =    initial crack densities
%           znzt =  crack zn/zt ratios (if method "beta" is used)
%           alpha = crack densities as a function of pressure
%           vin = input velocities
%           vbc = back-calc velocities using observed crack densities
%           vt = final fit velocities
%           
%
% Written by J.P. Verdon, University of Bristol, 2012
%
%

%% Read input parameters
% Choose method
switch lower(method)
    case 'scalar'
        iscal = 1;
        znzt=NaN;
    case 'beta'
        iscal = 2;
    otherwise
        error('Method is not recognised - must be scalar or beta')
end

% Deal with input velocities
if ischar(vdata)
    vfile=fopen(vdata);
    A=textscan(vfile,'%f%f%f%f%f%f%f%f','HeaderLines',0);
    for i=1:8
        if isfinite(A{i}(1))
            vin(:,i)=A{i};
        end
    end
else
    vin=vdata;
end

% Pressure data
P=vin(:,1)*1e6;
vin=vin(:,2:end);

% Determine whether we have isotropic or anisotropic data
switch size(vin,2)
    case 2
        iso = 1;
    case 7
        iso = 2;
    otherwise
        error('Input velocities must have 7 (triaxial) or 2 (isotropic) measurements')
end


% Deal with high pressure stiffness tensor
if ischar(cdata)
    if strcmpi(cdata,'high') == 1
        rho=2500;  % Density is not needed - set dummy value
        % Find highest velocity measurements
        [~,ind]=max(vin);
        imv=ceil(mean(ind));
        mv=vin(imv,:);
        cm=VK_vtoc(mv,rho);
        
    else
        cfile=fopen(cdata);
        A=textscan(cfile,'%f%f%f','HeaderLines',0);
        
        rho=A{3}(end);
        
        count=0;
        cm=NaN(6,6);
        for i=1:6
            for j=i:6
                count=count+1;
                cm(i,j)=A{3}(count)*rho;
            end
        end
        cm=VK_symmetry2d(cm,6);
    end
else
    cm=cdata;
    rho=2500;
end
        
% Check cm is a 6 by 6 tensor
if size(cm,1) ~= 6  &&  size(cm,2) ~= 6
    error('Input stiffness tensor is not 6 x 6')
end
        
% Get HP characteristics
[sm,~,num,mum,hm]=VK_stiffness_properties(cm);

%%





%% Compute crack density tensors
if iscal == 1
    [alpha,vbc]=VK_crack_density(P,vin,sm,hm,rho);
    if iso == 1 % Make alpha and vbc isotropic
        ma = mean(alpha,2);
        clear alpha
        alpha=ma;
        
        mvp=mean(vbc(:,1:4),2);
        mvs=mean(vbc(:,5:7),2);
        clear vbc
        vbc(:,1)=mvp;
        vbc(:,2)=mvs;
        
    end
        
elseif iscal == 2
    [alpha,znzt,vbc]=VK_crack_density_beta(P,vin,sm,hm,rho,iso);
    
    beta(:,1)=mean(alpha,2).*((znzt-1)');
    beta(:,2)=beta(:,1)/3;
end

%% 



%% Calculate misfit over crack density tensors
if iso == 1
    [asp,xi,xibc]=VK_cd_misfit(alpha,P,num,mum,1);
else
    [asp,xi,xibc]=VK_cd_misfit(alpha,P,num,mum,3);
end
    
%%



znzt_i = ones(size(znzt))*mean(znzt);
%% Calculate Tod velocites and quality control
if iscal == 1
    [vt,quality]=VK_velocity_tod(xibc,sm,hm,rho,vin,iso);
elseif iscal == 2
    [vt,quality]=VK_velocity_tod_beta(xibc,znzt_i,sm,hm,rho,vin,iso);
end


%%
        
colours='krgbmcy';
symbols='o^vsovs';
ls = {'-','--','-','--','-','--','-','--'};
slw = 1;
lw = 1.5;
msb = 10;
mss = 5;
fs = 14;

if strcmp(figures,'basic')
figure('Position',[680   531   328   447])
%subplot(2,2,[1 3])
set(gca,'FontSize',fs)
hold on
box on
if iso == 1
    n=2;
elseif iso == 2
    n=7;
end

for i=1:n
    plot(P/1e6,vin(:,i),[colours(i),symbols(i)],'MarkerSize',msb,'LineWidth',slw)
    %plot(P/1e6,vbc(:,i),[colours(i),symbols(i)],'MarkerSize',mss,'LineWidth',slw,'MarkerFaceColor',colours(i))
    plot(P/1e6,vt(:,i) ,[colours(i),ls{i}],'LineWidth',lw)
end
xlabel('Pressure (MPa)')
ylabel('Velocity (ms^{-1})')
xlim([min(P/1e6) max(P/1e6)])
end

if strcmp(figures,'all')
    figure('Position',[288   211   721   594])
    subplot(2,2,1)
    set(gca,'FontSize',fs)
    hold on ; box on ; grid on
  
    if iso == 1
        n=2;
    elseif iso == 2
        n=7;
    end
    for i=1:n
        plot(P/1e6,vin(:,i),[colours(i),symbols(i)],'MarkerSize',msb,'LineWidth',slw)
        plot(P/1e6,vbc(:,i),[colours(i),symbols(i)],'MarkerSize',mss,'LineWidth',slw,'MarkerFaceColor',colours(i))
    end
    ylabel('Velocity [m/s]')
    title('Back-calc vs observed V')
    xlim([min(P/1e6) max(P/1e6)])

    subplot(2,2,4)
    set(gca,'FontSize',fs)
    hold on ; box on ; grid on
    
    if iso == 1
        n=1;
    elseif iso == 2
        n=3;
    end

    for i=1:n
        plot(P/1e6,alpha(:,i),[colours(i),symbols(i)],'MarkerSize',msb,'LineWidth',slw)
        plot(P/1e6,xibc(:,i),colours(i),'LineWidth',lw)
    end
    xlabel('Pressure [MPa]')
    ylabel('Crack density')
    ylim([0 1])
    xlim([min(P/1e6) max(P/1e6)])
    title('Crack Density vs Pressure')

    if iscal == 2
    subplot(2,2,2)
        set(gca,'FontSize',fs)
    hold on ; box on ; grid on
    plot([min(P/1e6) max(P/1e6)],[1 1],'k--')
    plot(P/1e6,znzt,'ko-','MarkerSize',8,'LineWidth',2)
    set(gca,'XTickLabel',[])
    ylabel('Z_{N}/Z_{T}')
    xlim([min(P/1e6) max(P/1e6)])
    ylim([0 2])
    title('Z_{N}/Z_{T}')
    end

    subplot(2,2,3)
    set(gca,'FontSize',fs)
    hold on ; box on ; grid on
    if iso == 1
        n=2;
    elseif iso == 2
        n=7;
    end

    for i=1:n
        plot(P/1e6,vin(:,i),[colours(i),symbols(i)],'MarkerSize',msb,'LineWidth',slw)
        plot(P/1e6,vt(:,i) ,[colours(i),ls{i}],'LineWidth',lw)
    end
    xlabel('Pressure [MPa]')
    ylabel('Velocity [m/s]')
    xlim([min(P/1e6) max(P/1e6)])
    title('Final model velocities')
end


end







