function [sn,tau,rake,R] = RS_ResolveStress(S,az,inc)
%
% function [sn,tau,rake,R] = RS_ResolveStress(S,az,inc)
%
% This resolves the stress tensor onto a defined failure plane, in order to 
% determine whether failure has occured based on M-C criteria.
%
% Additionally, the resulting slip vector is converted into a rake angle.
% Finally, the rotation matrix to move from global into the fault-slip
% coordinate frame is returned.
%
% Stress on the fault plane is resolved using the equations defined in
% http://www.geofys.uu.se/bl/Avh/node7.html
% and the fault-slip coord system is based on McKenzie, BSSA, 1969.
%
% The fault slip plane is defined by 3 orthogonal vectors:
%    1 = direction of slip
%    2 = fault normal direction
%    3 = 1 x 2 (in fault plane, perpendicular to direction of motion)
%
% INPUTS:   S = stress tensor
%           az = fault plane azimuth
%           inc = fault plane dip
%
% OUTPUTS:  sn = normal stress
%           tau = shear stress
%           rake = rake angle of fault slip
%           R = fault-slip coordinate system rotation tensor
%
%
% Written by J.P. Verdon, 2014
% University of Bristol
%



% Generate fault normal vector
i=sind(az)*sind(inc);
j=cosd(az)*sind(inc);
k=cosd(inc);
n=[i ; j ; k];

t = -(S*n); % Traction vector

sn_v = dot(t,n)*n; % Fault normal stress vector

tau_v = -(t - dot(t,n)*n); % Shear stress vector

sn = norm(sn_v); % Normal stress is magnitude of normal vector
tau = norm(tau_v);  % Shear stress is magnitude of shear vector

% Determine whether normal stress is tensional or extensional
if dot(n,sn_v) > 0
    sn = -sn;
end


% Define a horizontal lineation along the fault plane, from which rake is
% measured
hl = [sind(az+90) ; cosd(az+90) ; 0]; 

rake = acosd(dot(hl,tau_v/tau));

% Create a normalised vector defining the fault-slip coordinate system
tnorm=tau_v/tau;
snorm=sn_v/norm(sn_v);
pnorm = cross(tnorm,snorm);
R = [tnorm snorm pnorm];



