% function CriticalStressBall(S,...)
%
% Generate a plot of normal and shear stress, and the M-C criteria,
% C = mu*sn + coh - tau
%
% Input argument is stress tensor S
%
% Optional input are mu and coh, which default to 0.6 and 0
%


function CriticalStressBall(S,varargin)

% Default arguments
mu = 0.6;
coh = 0;
nc = 20;
fignum = 'new';

% Process the optional arguments (overwriting defaults where defined)
iarg = 1;
while iarg <= (length(varargin))
    switch lower(varargin{iarg})
        case 'mu'
            mu = varargin{iarg+1};
            iarg = iarg + 2;
            
        case 'coh'
            coh = varargin{iarg+1};
            iarg = iarg + 2;
            
        case 'nc'
            nc = varargin{iarg+1};
            iarg = iarg + 2;
            
        case 'fignum'
            fignum = varargin{iarg+1};
            iarg = iarg + 2;
            
        otherwise
            error('BS:UnknownOption',...
                ['Unknown option: ',varargin{iarg}]);
    end
end

% Create a mesh of azimuth and inclination
inc = 0:2:90;
az = 0:2:360;
[X,Y]=meshgrid(az,inc);
az=X(:);
inc=Y(:);
n = length(az);

% Loop over all values, and do calculations
sn = NaN(n,1);
t = NaN(n,1);
for m = 1:n
    [sn(m),t(m)] = RS_ResolveStress(S,az(m),inc(m));    
end

% Reshape result vectors for contour plotting
snp = reshape(sn,size(X));
tp = reshape(t,size(X));

% Calculate M-C criteria
Cr = tp - mu.*snp - coh;

% Convert az and inc into plotting coords
[X,Y,~] = pol2cart((90-X)*pi/180,Y/90);

% Generate figure
if strcmp(fignum,'new');
    figure('Position',[150 560 1342 394]);
else
    figure(fignum);
    set(gcf,'Position',[150 560 1342 394]);
end
clf
subplot('Position',[0.05 0.1 0.266 0.8],'FontSize',16)
set(gca,'DataAspectRatio',[1 1 1])
hold on
title('Effective Normal Stress')

h=polar([0 2*pi],[0 max(max(X))]);
delete(h)

contourf(X,Y,snp/1e6,nc,'EdgeColor','none');
view(90,-90)
[P,Q,~] = sph2cart(60*pi/180,(90-90)*pi/180,1);

set(gca,'Color','none','XTick',[],'YTick',[],'XColor','w','YColor','w')
colorbar('FontSize',16)


subplot('Position',[0.366 0.1 0.266 0.8])
set(gca,'DataAspectRatio',[1 1 1],'FontSize',16)
hold on
title('Effective Shear Stress')

h=polar([0 2*pi],[0 max(max(X))]);
delete(h)

contourf(X,Y,tp/1e6,nc,'EdgeColor','none');
view(90,-90)

set(gca,'Color','none','XTick',[],'YTick',[],'XColor','w','YColor','w')
colorbar('FontSize',16)


subplot('Position',[0.6833 0.1 0.266 0.8])
set(gca,'DataAspectRatio',[1 1 1],'FontSize',16)
hold on
title('M-C Criteria')

h=polar([0 2*pi],[0 max(max(X))]);
delete(h)

contourf(X,Y,Cr/1e6,nc,'EdgeColor','none');
view(90,-90)

set(gca,'Color','none','XTick',[],'YTick',[],'XColor','w','YColor','w')
colorbar('FontSize',16)
