% function PredictMmaxPlot(M,T,MmaxHallo,MmaxShap,Seff,SI,b,time,...)
%
% Generate a plot of Mmax vs time against observed magnitudes
%
% INPUTS:   M = observed magnitudes
%           T = observed event times
%           MmaxHallo = modelled Mmax as a function of time using Hallo method
%           MmaxShap = modelled Mmax as a function of time using Shapiro method 
%           Seff = seismic efficiency as a function of time
%           SI = seismogenic index as a function of time
%           b = b-value as a function of time
%           time = time axis
%
% OPTIONAL INPUTS:
%           datetick = logical to use date-tick axes
%           colors = plotting colors in order of Mmax, Seff and b lines
%               default = {'r','b','m'};
%           linestyles = styles of lines to be plotted
%               default = {'-','-','-'}
%           xlim = x limits
%               default = min and max of time axis
%           maglim = magnitude limit in plots
%           blim = limits of b in plot
%           seffscale = scale of Seff plot - lin or log
%           bdir = direction of b-value plot (positive up or down)
%
%
% Written by J.P. Verdon,
% Outer Limits Geophysics LLP
% Bristol, 08/2015
% www.OLGeo.co.uk
% James@OLGeo.co.uk
%


function PredictMmaxPlot(M,T,MmaxH,MmaxS,Seff,SI,b,time,varargin)

dt = false;
dttag = 'HH';
maglim = NaN;
sefflim = NaN;
blim = [0 4];
seffscale = 'log';
bdir = 'rev';
tstart = 'relative';
timeunits = 'days';

% Check that time is a row vector
if ~isrow(time)
    time = time';
end
colors = {'r','b','m',[1 0.65 0],'g'};
linestyles = {'-','-','-','-','-'};

% Process the optional arguments (overwriting defaults where defined)
iarg = 1;
while iarg <= (length(varargin))
    switch lower(varargin{iarg})
        case 'datetick'
            dt = varargin{iarg+1};
            iarg = iarg + 2; 
        
        case 'dttag'
            dttag = varargin{iarg+1};
            iarg = iarg + 2;
            
        case 'linestyles'
            iarg = iarg + 2;
        
        case 'colors'
            iarg = iarg + 2;
            
        case 'xlim'
            xl = varargin{iarg+1};
            iarg = iarg + 2;
            
        case 'maglim'
            maglim = varargin{iarg+1};
            iarg = iarg + 2;
        
        case 'sefflim'
            sefflim = varargin{iarg+1};
            iarg = iarg + 2;
            
        case 'blim'
            blim = varargin{iarg+1};
            iarg = iarg + 2;
        
        case 'seffscale'
            seffscale = varargin{iarg+1};
            iarg = iarg + 2;
            
        case 'bdir'
            bdir = varargin{iarg+1};
            iarg = iarg + 2;
            
        case 'tstart'
            tstart = varargin{iarg+1};
            iarg = iarg + 2;
            
        case 'timeunits'
            timeunits = varargin{iarg+1};
            iarg = iarg + 2;
            
        otherwise
            error('BS:UnknownOption',...
                ['Unknown option: ',varargin{iarg}]);
    end
end

% Make sure data for multiple stages are in columns, not rows
Seff = row2col(Seff);
SI = row2col(SI);
b = row2col(b);
time = row2col(time);
MmaxH = row2col(MmaxH);
MmaxS = row2col(MmaxS);
M = row2col(M);
T = row2col(T);


% Log10 of Seff is asked
if strcmp(seffscale,'log')
    Seff = log10(Seff);
end
            
% Adjust event times to relative to injection start if asked 
if strcmp(tstart,'relative')
    T = T - time(1);
    time = time-time(1);
    
    if strcmp(timeunits,'hours')
        T = T*24;
        time = time*24;
    end
    if ~exist('xl','var')
        xl = [0 max(time)];
    end
else
    if ~exist('xl','var')
        xl = minmax(time');
    end
end


clf
subplot('Position',[0.1 0.43 0.8 0.47],'FontSize',16)
hold on
box on
grid on

ok = (isfinite(MmaxH));
ar1 = plot(time(ok),MmaxH(ok),linestyles{1},'Color',colors{1},'LineWidth',1.5);
ar3 = plot(T,M,['k','o']);
ok = (isfinite(MmaxS));
ar2 = plot(time(ok),MmaxS(ok),linestyles{2},'Color',colors{2},'LineWidth',1.5);

legend([ar1,ar2,ar3],{'M_{Max} (Hallo method)','M_{Max} (Shapiro method)','Observed Events'},'Location','best','Box','off');

set(gca,'XAxisLocation','top')
xlim(xl)
if isfinite(maglim)
    ylim(maglim)
end
if dt
    datetick('x',dttag,'keeplimits')
end
ylabel('Magnitude')

subplot('Position',[0.1 0.09 0.8 0.3],'FontSize',16,'YColor',colors{3})
hold on
box on
grid on
ok = (isfinite(Seff));
ar1 = plot(time(ok),Seff(ok),'LineStyle',linestyles{3},'Color',colors{3},'LineWidth',1.5);
ok = (isfinite(SI));
ar3 = plot(time(ok),SI(ok),'LineStyle',linestyles{4},'Color',colors{4},'LineWidth',1.5);

xlim(xl)
if isfinite(sefflim)
    ylim(sefflim);
end
if dt
    datetick('x',dttag,'keeplimits')
end
if strcmp(seffscale,'log')
    ylabel('log_{10}(S_{EFF}) and \Sigma')
else
    ylabel('S_{EFF}')
end
    
pos = get(gca,'Position');

ax2 = axes('Position',pos,'XAxisLocation','bottom','YAxisLocation','right',...
    'Color','none','XColor','k','YColor',colors{5},'XLim',xl,...
    'FontSize',16,'XTick',[],'YLim',blim,'YDir',bdir);

ok = (isfinite(b));
ar2 = line(time(ok),b(ok),'Color',colors{5},'Parent',ax2,'LineWidth',1.5,...
        'LineStyle',linestyles{5});
set(get(ax2,'YLabel'),'String','b value','FontSize',16)

legend([ar2,ar1,ar3],{'b','S_{EFF}','\Sigma'},'Location','best','Box','off');
xlabel([char(10),'Time from Injection Start (hours)'])


end




function Y = row2col(X)

if size(X,1) < size(X,2)
    Y = X';
else
    Y = X;
end
end










