% function [Mmax1,Mmax2,Seff,b,SI,MC,IC] = PredictMmax(M,T,IR,IT,Vmax,...)
%
%
% Mmax = PredictMmax(M,T,IR,IT,'G','20e9,'nmin',20,'figures',true);
%
%
% Predict the maximum magnitude if injection continues, based on injection
% volumes and seismicity observed thus far. See Verdon and Budge (2018, BSSA)
% for more details
%
% INPUTS:   M = magnitudes of events observed thus far in an operation
%           T = occurence times of said events
%           IR = injection rate during operation thus far
%           IT = times corresponding to said injection rates
%           Vmax = intended final injection volume
%
% OPTIONAL INPUTS:
%           G = Young's modulus
%           delta = Hallo (2014) delta value
%           chi = Shapiro (2010) chi probability exceedance value
%           E = ks-value for B value computation
%           nmin = minimum number of events needed for b value computation
%           rounded = true/false, is Mag data rounded to 0.1 bins, or
%               continuous
%           defaultb = default b value to use if a b value cannot be
%               computed
%           mminmethod = method for assigning the Mmin value when computing
%               b values
%           mminmax = maximum allowable Mmin value
%
%
% OUTPUTS:  Mmax1 = maximum event magnitude to be triggered by the end of
%               operation at Vmax (using Hallo et al. (2014) method)
%           Mmax2 = maximum event magnitude to be triggered by the end of
%               operation at Vmax (using Shapiro et al. (2010) method)
%           Seff = seismic efficiency
%           b = b value
%           SI = seismogenic index
%           MC = cumulative moment release 
%           IC = cumulative injection volume
%
% Written by J.P. Verdon,
% University of Bristol
%


function [Mmax1,Mmax2,Seff,b,SI,MC,IC] = PredictMmax(M,T,IR,IT,Vmax,varargin)

% Default values for optional variables
G = 20e9;
delta = 0.1448;
E = 0.2;
nmin = 35;
Msearch = max(M)+1;
rounded = false;
defaultb = 1.5;
mminmethod = 'ks';
defaultmmin = NaN;
mminmax = NaN;
chi = 0.95;

% Process the optional arguments (overwriting defaults where defined)
iarg = 1;
while iarg <= (length(varargin))
    switch lower(varargin{iarg})
        case 'bins'
            bins = varargin{iarg+1};
            iarg = iarg + 2;
        
        case 'g'
            G = varargin{iarg+1};
            iarg = iarg + 2;
            
        case 'delta'
            delta = varargin{iarg+1};
            iarg = iarg + 2;
            
        case 'e'
            E = varargin{iarg+1};
            iarg = iarg + 2;
            
        case 'nmin'
            nmin = varargin{iarg+1};
            iarg = iarg + 2;
            
        case 'rounded'
            rounded = varargin{iarg+1};
            iarg = iarg + 2;
            
        case 'msearch'
            Msearch = varargin{iarg+1};
            iarg = iarg + 2;
        
        case 'defaultb'
            defaultb = varargin{iarg+1};
            iarg = iarg + 2;
        
        case 'mminmethod'
            mminmethod = varargin{iarg+1};
            iarg = iarg + 2;
            
        case 'mminmax'
            mminmax = varargin{iarg+1};
            iarg = iarg + 2;
        
        case 'chi'
            chi = varargin{iarg+1};
            iarg = iarg + 2;

        otherwise
            error('PredictMmax:UnknownOption',...
                ['Unknown option: ',varargin{iarg}]);
    end
end



% Compute the B value and the minimum magnitude of completeness
if length(M) < nmin
    Mmax1 = NaN;
    Mmax2 = NaN;
    Seff = NaN;
    SI = NaN;
    b = NaN;
    MC = NaN;
    IC = NaN;
    return
end
if ~exist('bins','var')
    bins = sort(M);
end
[b,~,~,Mmin] = BValue(M,'E',E,'nmin',nmin,'figure',false,'rounded',rounded,'bins',bins,'mminmethod',mminmethod);
if Mmin > mminmax
    [b,~,~,Mmin] = BValue(M,'E',E,'nmin',nmin,'figure',false,'rounded',rounded,'bins',bins,'mminmethod',mminmax);
end

if isnan(Mmin)
    Mmin = defaultmmin;
    %warning('No Mmin found');
    Mmax1 = NaN;
    Mmax2 = NaN;
    Seff = NaN;
    SI = NaN;
    b = NaN;
    MC = NaN;
    IC = NaN;
    return
end

% Consider events above Mmin only
Mcut = M(M>=Mmin);
Tcut = T(M>=Mmin);

% Convert catalogue magnitudes to moment
Mo = 10.^(1.5*Mcut + 9.1);

% Bin moments so that injection rates and moment sums are on same time axis
Mbins = zeros(length(IT),1);
for i = 2:length(IT)
    Mcut = Mo(Tcut<=IT(i) & Tcut>IT(i-1));
    Mbins(i) = sum(Mcut);
end

% Cumulative injection volumes and injection rates
MC = cumsum(Mbins);
IC = cumsum(IR);

% Compute seismic efficiency at end of injection
Seff = MC(end)/(G*IC(end));

% Default b value if not enough events to compute b
if isnan(b)
    b = defaultb;
end


% Compute predicted cumulative moment at the end of injection
MoPred = Vmax*Seff*G;
% Search function for moment sum as function of maximum magnitude
SMO = @(MwMax) b*(10^(b*MwMax-log10((10^(b*delta))-10^(-b*delta))))*(10^9.1)*(10^(MwMax*(1.5-b)) - 10^(Mmin*(1.5-b)))/(1.5-b);
% Solve to find maximum magnitude that matches cumulative moment
CF = @(x) abs(SMO(x) - MoPred(end)); 
Mmax1 = fminsearch(CF,Msearch,optimset('display','none'));

Mmax1 = Mmax1 + 0.5; % 0.5 unit increase to ensure that 95% certainty is exceeded

MC = MC(end);
IC = IC(end);

% Shapiro Mmax method
MLc=M(M>Mmin); % Use only events larger than the Mmin value
NE=length(MLc);
SI = log10(NE)-log10(IC)+b*Mmin;
Mmax2 = (SI-log10((-1./IC).*log(chi)))./b;




