% function INSAFF_SphereFromFortran(of)
% 
% Function to generate a hemisphere plot displaying the results of an
% INSAFF inversion.
%
% This routine is designed to be called from within the fortran executable.
% It is not run from the matlab command line, and will be default quit the
% instance of matlab from which it has been run. 
%
% The input "of" is a string giving the stem of the output files created by
% INSAFF. 
%
% Dependencies: requires that the MSAT toolbox is installed
% http://www1.gly.bris.ac.uk/MSAT/
%
% Also requires "export_fig"
% https://uk.mathworks.com/matlabcentral/fileexchange/23629-export-fig
%
% Written by J.P. Verdon, University of Bristol, 2015
%



function INSAFF_SphereFromFortran(of)

% Check that file exists
ierror = 0;
if ~exist([of,'.out'],'file');
    ierror = 1;
    fprintf('%s \n',['Error: File ',[of,'.out'],' has not been created']); 
end
if ~exist('MS_plot');
    ierror = 1;
    fprintf('%s \n','Error: MSAT toolbox not found. MSAT can be downloaded from: http://www1.gly.bris.ac.uk/MSAT/'); 
end


if ierror == 0
    % Read in parameters from the output file
    fprintf('%s \n',of)
    
    fid = fopen([of,'.out']);
    nl = linecount(fid);
    for i = 1:nl
        tline = fgetl(fid);
        if tline ~=-1
            A = textscan(tline,'%s',1);
            if length(A{1}{1})>4 && strcmp(A{1}{1}(1:4),'CIJ=');
                cijfile = A{1}{1}(5:end);
            end
            if length(A{1}{1})>5 && strcmp(A{1}{1}(1:5),'DATA=');
                dfile = A{1}{1}(6:end);
            end
            if length(A{1}{1})>3 && strcmp(A{1}{1}(1:4),'RHO=');
                B = textscan(tline,'%*s%f',1);
                rho = B{1};
            end
            if length(A{1}{1})>6 && strcmp(A{1}{1}(1:6),'SETUP=');
                setup = A{1}{1}(7:end);
            end
            if length(A{1}{1})>6 && strcmp(A{1}{1}(1:6),'CFILE=');
                cfile = A{1}{1}(7:end);
            end
        end
    end
    fclose(fid);

    % Get control parameters from the control file
    t = fileread(cfile);
    t(t==char(10)) = ' ';  % Change newlines for spaces for string finding
    ts = str2num(get_string(t,'poltickscale'));
    if isempty(ts)
        ts = 1/200;
    end
    cmap = get_string(t,'cmap');
    if isempty(cmap)
        cmap = 'parula';
    end

    % Get the stiffness tensor from the cij file
    C = load(cijfile);

    % Get the observations from the datafile
    D = load(dfile);

    
    if strcmp(setup,'surface')
        MS_plot(C/1e9,rho,'sdata',D(:,1)',D(:,2)',D(:,3)'-D(:,1)',D(:,4)',...
            'plotmap',{'AVSPOL'},'cmap',cmap,'poltickscale',ts,...
            'titletext',{of},'reverse');
    elseif strcmp(setup,'downhole')
        MS_plot(C/1e9,rho,'sdata',D(:,1)',D(:,2)',D(:,3)',D(:,4)',...
            'plotmap',{'AVSPOL'},'cmap',cmap,'poltickscale',ts,...
            'titletext',{of},'reverse');
    else
        error(['Error: Setup type ',setup,' not recognised'])
    end
    set(gcf,'Position',[1 1 951 805])
    export_fig([of,'.png'], '-png','-transparent','-r100');

    close(gcf)
end
exit



end