!***************************************************************************************************
!
!   tensors_lib.f90
!      contains subroutines to manipulate tensors
!
!   Written by J.P. Verdon, University of Bristol, 2011
!      based partly on legacy f77 code from:
!							   J-M. Kendall, UoB, 2005-2011 (and Leeds before that)
!							   D.A. Angus, UoB, 2005-2010
!      and f90 libraries from:
! 							   J. Wookey, UoB, 2005-2011
!
!-------------------------------------------------------------------------------
!  This software is distributed in the hope that it will be useful, but WITHOUT
!  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
!  FITNESS FOR A PARTICULAR PURPOSE.
!
!  This software is distributed for ACADEMIC PURPOSES and is not to be 
!  distributed to any other parties.  This code is NOT TO BE INTEGRATED 
!  within or developed into COMMERCIAL CODE unless WRITTEN PERMISSION 
!  is given by the authors.
!-------------------------------------------------------------------------------
!
!     Last edited: March 2011
!
!***************************************************************************************************


!***************************************************************************************************
   subroutine get_s_nus_mus_and_es(c,s,ae,anu,amu,h)
!***************************************************************************************************
implicit none
integer						:: i
real,dimension(6,6)			:: c,s
real,dimension(3)			:: ae,anu,amu,h

call invmat(c,6,s)    ! Invert for compliance
do i=1,3     ! Young's modulus
	ae(i)=1/s(i,i)
enddo
anu(1)=ae(1)*(-s(1,2)-s(1,3))/2. ! Poisson's ratio
anu(2)=ae(2)*(-s(2,1)-s(2,3))/2.
anu(3)=ae(3)*(-s(3,1)-s(3,2))/2.
 
amu(1)=(c(6,6)+c(5,5))/2.    ! Average shear modulus
amu(2)=(c(6,6)+c(4,4))/2.
amu(3)=(c(5,5)+c(4,4))/2.

do i=1,3     ! Gueguen 'h' factor
	h(i)=3.*ae(i)*(2.-anu(i))/(32.*(1.-anu(i)*anu(i)))
enddo

return
end subroutine get_s_nus_mus_and_es
!***************************************************************************************************


!***************************************************************************************************
   subroutine cfromv(c,vp,vs,rho)
!***************************************************************************************************
implicit none
integer 	:: i
real		:: c(6,6),vp,vs,rho

call zeros2d(c,6,6)
c(1,1)=vp*vp*rho
do i=2,3
    c(i,i)=c(1,1)
enddo
c(4,4)=vs*vs*rho
do i=5,6
    c(i,i)=c(4,4)
enddo
c(1,2)=c(1,1)-2*c(4,4)
c(1,3)=c(1,2)
c(2,3)=c(1,2)
call symmetry2d(c,6)
return
end subroutine cfromv
!***************************************************************************************************


!***************************************************************************************************
   subroutine isotropic(c)
!***************************************************************************************************
implicit none
real			:: c(6,6),k,mu
integer			:: i,j

k=(c(1,1)+c(2,2)+c(3,3)+2.0*(c(1,2)+c(2,3)+c(1,3)))/9.0
mu=(c(1,1)+c(2,2)+c(3,3)-c(1,2)-c(2,3)-c(1,3)+3.0*(c(4,4)+c(5,5)+c(6,6)))/15.0
call zeros2d(c,6,6)
c(1,1)=k+4*mu/3
do i=2,3
	c(i,i)=c(1,1)
enddo
do i=4,6
	c(i,i)=mu
enddo
c(1,2)=c(1,1)-2*c(4,4)
c(1,3)=c(1,2)
c(2,3)=c(1,2)
call symmetry2d(c,6)
return
end subroutine isotropic
!***************************************************************************************************


!***************************************************************************************************
   subroutine atrakto6by6(itype,c,ec)
!***************************************************************************************************
implicit none
integer 		:: itype,i,j,k
real			:: c(6,6),ec(21)

if (itype.eq.0)then  ! From ATRAK to 6x6
	k=0
	do i=1,6
		do j=i,6
			k=k+1
	    	c(i,j)=ec(k)
		enddo
	enddo
	call symmetry2d(c,6)
elseif (itype.eq.1)then  ! From 6x6 to ATRAK
	k=0
	do i=1,6
		do j=i,6
			k=k+1
			ec(k)=c(i,j)
		enddo
	enddo
endif
return
end subroutine atrakto6by6
!***************************************************************************************************

!***************************************************************************************************
   subroutine rotatelc(a,cc,cct)
!***************************************************************************************************
implicit none
integer			:: i,j,k,l,m,n,r,s
real			:: csum,cc(3,3,3,3),cct(3,3,3,3),a(3,3)

do m=1,3
	do n=1,3
    	do r=1,3
        	do s=1,3
            	csum=0.
                do i=1,3
                	do j=1,3
                    	do k=1,3
                        	do l=1,3
                csum=csum+a(m,i)*a(n,j)*a(r,k)*a(s,l)*cc(i,j,k,l)
                            enddo
                		enddo
                	enddo
            	enddo
                cct(m,n,r,s)=csum
            enddo
        enddo
    enddo
enddo

return
end subroutine rotatelc
!***************************************************************************************************

      
!***************************************************************************************************
   subroutine invmat(cin,n,cout)
!***************************************************************************************************
implicit none
integer 				:: i,j,n,indx(n)
real					:: cin(n,n),cout(n,n),ctemp(n,n),d

do i=1,n
	do j=1,n
		ctemp(i,j)=cin(i,j)
		cout(i,j)=0.0
	enddo
	cout(i,i)=1.0
enddo
call ludcmp(ctemp,n,n,indx,d)
do j=1,n
	call lubksb(ctemp,n,n,indx,cout(1,j))
enddo
return
end subroutine invmat
!***************************************************************************************************


!***************************************************************************************************
   subroutine s_ijkl(a6,a3)
! Returns the s_ijkl given the s_mn Voigt notation for the 
! COMPLIANCE (not stiffness) tensor. 
!***************************************************************************************************
implicit none
integer			:: i,j,k,l
real			:: a6(6,6),a3(3,3,3,3)

call zeros4d(a3,3,3,3,3)

a3(1,1,1,1)=a6(1,1)
a3(2,2,2,2)=a6(2,2)
a3(3,3,3,3)=a6(3,3)
a3(1,1,2,2)=a6(1,2)
a3(1,1,3,3)=a6(1,3)
a3(2,2,3,3)=a6(2,3)
  
a3(1,2,1,2)=a6(6,6)/4
a3(1,3,1,3)=a6(5,5)/4
a3(2,3,2,3)=a6(4,4)/4
 
a3(1,1,2,3)=a6(1,4)/2
a3(1,1,1,3)=a6(1,5)/2
a3(1,1,1,2)=a6(1,6)/2
 
a3(2,2,2,3)=a6(2,4)/2
a3(2,2,1,3)=a6(2,5)/2
a3(2,2,1,2)=a6(2,6)/2

a3(3,3,2,3)=a6(3,4)/2
a3(3,3,1,3)=a6(3,5)/2
a3(3,3,1,2)=a6(3,6)/2
 
a3(2,3,1,3)=a6(4,5)/4
a3(2,3,1,2)=a6(4,6)/4

a3(1,3,1,2)=a6(5,6)/4

call symmetry4d(a3,3)

return
end subroutine s_ijkl
!***************************************************************************************************
      

!***************************************************************************************************
  subroutine s_ij(a3,a6)
!  Returns the s_ij (Voigt) given the s_ijkl for the COMPLIANCE 
!  not stiffness) tensor  
!***************************************************************************************************
implicit none
real		:: a6(6,6),a3(3,3,3,3)
	  
call zeros2d(a6,6,6)
a6(1,1)=a3(1,1,1,1)
a6(2,2)=a3(2,2,2,2)
a6(3,3)=a3(3,3,3,3)
a6(1,2)=a3(1,1,2,2)
a6(1,3)=a3(1,1,3,3)
a6(2,3)=a3(2,2,3,3)  
a6(6,6)=4*a3(1,2,1,2)
a6(5,5)=4*a3(1,3,1,3)
a6(4,4)=4*a3(2,3,2,3) 
a6(1,4)=2*a3(1,1,2,3)
a6(1,5)=2*a3(1,1,1,3)
a6(1,6)=2*a3(1,1,1,2) 
a6(2,4)=2*a3(2,2,2,3)
a6(2,5)=2*a3(2,2,1,3)
a6(2,6)=2*a3(2,2,1,2)
a6(3,4)=2*a3(3,3,2,3)
a6(3,5)=2*a3(3,3,1,3)
a6(3,6)=2*a3(3,3,1,2)
a6(4,5)=4*a3(2,3,1,3)
a6(4,6)=4*a3(2,3,1,2)
a6(5,6)=4*a3(1,3,1,2)

call symmetry2d(a6,6)

return
end subroutine s_ij
!***************************************************************************************************


!***************************************************************************************************
  subroutine a_ijkl(a6,a3)
!***************************************************************************************************
implicit none
real			:: a6(6,6),a3(3,3,3,3)

a3(1,1,1,1)=a6(1,1)
a3(2,2,2,2)=a6(2,2)
a3(3,3,3,3)=a6(3,3)
a3(1,1,2,2)=a6(1,2)
a3(1,1,3,3)=a6(1,3)
a3(2,2,3,3)=a6(2,3)
a3(1,2,1,2)=a6(6,6)
a3(1,3,1,3)=a6(5,5)
a3(2,3,2,3)=a6(4,4)
a3(1,1,2,3)=a6(1,4)
a3(1,1,1,3)=a6(1,5)
a3(1,1,1,2)=a6(1,6)
a3(2,2,2,3)=a6(2,4)
a3(2,2,1,3)=a6(2,5)
a3(2,2,1,2)=a6(2,6)
a3(3,3,2,3)=a6(3,4)
a3(3,3,1,3)=a6(3,5)
a3(3,3,1,2)=a6(3,6)
a3(2,3,1,3)=a6(4,5)
a3(2,3,1,2)=a6(4,6)
a3(1,3,1,2)=a6(5,6)
call symmetry4d(a3,3)

return
end subroutine a_ijkl
!***************************************************************************************************


!***************************************************************************************************
  subroutine a_ij(a3,a6)
!***************************************************************************************************
implicit none
real			:: a3(3,3,3,3),a6(6,6)

call zeros2d(a6,6,6)
a6(1,1)=a3(1,1,1,1)
a6(2,2)=a3(2,2,2,2)
a6(3,3)=a3(3,3,3,3)
a6(1,2)=a3(1,1,2,2)
a6(1,3)=a3(1,1,3,3)
a6(2,3)=a3(2,2,3,3)
a6(6,6)=a3(1,2,1,2)
a6(5,5)=a3(1,3,1,3)
a6(4,4)=a3(2,3,2,3)
a6(1,4)=a3(1,1,2,3)
a6(1,5)=a3(1,1,1,3)
a6(1,6)=a3(1,1,1,2)
a6(2,4)=a3(2,2,2,3)
a6(2,5)=a3(2,2,1,3)
a6(2,6)=a3(2,2,1,2)
a6(3,4)=a3(3,3,2,3)
a6(3,5)=a3(3,3,1,3)
a6(3,6)=a3(3,3,1,2)
a6(4,5)=a3(2,3,1,3)
a6(4,6)=a3(2,3,1,2)
a6(5,6)=a3(1,3,1,2)
call symmetry2d(a6,6)

return
end subroutine a_ij
!***************************************************************************************************

      
!***************************************************************************************************
  subroutine transpose(a,b,n,m)
!  Finds b, which is the transpose of matrix a, which is n by m
!***************************************************************************************************
implicit none
integer i,j,n,m
real a(3,3),b(3,3)
	  
do i=1,n
	do j=1,m
		b(j,i)=a(i,j)
	enddo
enddo 
return
end subroutine transpose
!***************************************************************************************************


!***************************************************************************************************
  subroutine zeros2d(a,n1,n2)
!***************************************************************************************************
implicit none
integer 			:: i,j,n1,n2
real 				:: a(n1,n2)
	  
do i=1,n1
	do j=1,n2
		a(i,j)=0.0
	enddo
enddo
return
end subroutine zeros2d
!***************************************************************************************************
	 	  

!***************************************************************************************************
  subroutine zeros(a,n)
!***************************************************************************************************
implicit none
integer 			:: i,n
real 				:: a(n) 
do i=1,n
	a(i)=0.0
enddo
return
end subroutine zeros
!***************************************************************************************************

	  
!***************************************************************************************************
  subroutine zeros4d(a,n1,n2,n3,n4)
!***************************************************************************************************
implicit none
integer 			:: i,j,k,l,n1,n2,n3,n4
real 				:: a(n1,n2,n3,n4)
	  
do i=1,n1
	do j=1,n2
		do k=1,n3
			do l=1,n4
				a(i,j,k,l)=0.
			enddo
		enddo
	enddo
enddo
return
end subroutine zeros4d
!***************************************************************************************************


!***************************************************************************************************
  subroutine stressrot(estress,sigr,eval)!eval)
!***************************************************************************************************
implicit none
integer 			:: i,j
real				:: evalue
real,dimension(3,3) :: estress(3,3),sigr(3,3),sigt(3,3)
real,dimension(3)	:: drota(3),eval(3),erot(3)

! Evaluate change of basis tensor to rotate elasticity tensor into 
! triaxial stress field principal components
do i=1,3
	do j=1,3
    	sigt(i,j)=estress(i,j)
    enddo
enddo
call tred(sigt,3,3,drota,erot)
call tqli(drota,erot,3,3,sigt)
do i=1,3
	sigr(i,1)=sigt(i,1)
    sigr(i,2)=sigt(i,2)
    sigr(i,3)=sigt(i,3)
enddo
eval(1)=drota(1)
eval(2)=drota(2)
eval(3)=drota(3)
!      evalue=-eval(3)
return
end subroutine stressrot
!***************************************************************************************************


!***************************************************************************************************
  subroutine symmetry2d(a,n)
!***************************************************************************************************
implicit none
integer			:: n,i,j
real			:: a(n,n)

do i=1,n
	do j=i,n
    	if(i.ne.j)a(j,i)=a(i,j)
    enddo
enddo
return
end subroutine symmetry2d
!***************************************************************************************************


!***************************************************************************************************
  subroutine symmetry4d(a,n)
!***************************************************************************************************
implicit none
integer			:: n
real			:: a(n,n,n,n)

a(2,2,1,1)=a(1,1,2,2)
a(3,3,1,1)=a(1,1,3,3)
a(3,3,2,2)=a(2,2,3,3)
a(2,1,1,2)=a(1,2,1,2)
a(1,2,2,1)=a(1,2,1,2)
a(2,1,2,1)=a(1,2,1,2)
a(3,1,1,3)=a(1,3,1,3)
a(1,3,3,1)=a(1,3,1,3)
a(3,1,3,1)=a(1,3,1,3)
a(3,2,2,3)=a(2,3,2,3)
a(2,3,3,2)=a(2,3,2,3)
a(3,2,3,2)=a(2,3,2,3)
a(1,1,3,2)=a(1,1,2,3)
a(2,3,1,1)=a(1,1,2,3)
a(3,2,1,1)=a(1,1,2,3)
a(1,1,3,1)=a(1,1,1,3)
a(1,3,1,1)=a(1,1,1,3)
a(3,1,1,1)=a(1,1,1,3)
a(1,1,2,1)=a(1,1,1,2)
a(1,2,1,1)=a(1,1,1,2)
a(2,1,1,1)=a(1,1,1,2)
a(2,2,3,2)=a(2,2,2,3)
a(2,3,2,2)=a(2,2,2,3)
a(3,2,2,2)=a(2,2,2,3)
a(2,2,3,1)=a(2,2,1,3)
a(1,3,2,2)=a(2,2,1,3)
a(3,1,2,2)=a(2,2,1,3)
a(2,2,2,1)=a(2,2,1,2)
a(1,2,2,2)=a(2,2,1,2)
a(2,1,2,2)=a(2,2,1,2)
a(3,3,3,2)=a(3,3,2,3)
a(2,3,3,3)=a(3,3,2,3)
a(3,2,3,3)=a(3,3,2,3)
a(3,3,3,1)=a(3,3,1,3)
a(1,3,3,3)=a(3,3,1,3)
a(3,1,3,3)=a(3,3,1,3)
a(3,3,2,1)=a(3,3,1,2)
a(1,2,3,3)=a(3,3,1,2)
a(2,1,3,3)=a(3,3,1,2)
a(2,3,3,1)=a(2,3,1,3)
a(3,2,1,3)=a(2,3,1,3)
a(3,2,3,1)=a(2,3,1,3)
a(1,3,2,3)=a(2,3,1,3)
a(1,3,3,2)=a(2,3,1,3)
a(3,1,2,3)=a(2,3,1,3)
a(3,1,3,2)=a(2,3,1,3)
a(2,3,2,1)=a(2,3,1,2)
a(3,2,1,2)=a(2,3,1,2)
a(3,2,2,1)=a(2,3,1,2)
a(1,2,2,3)=a(2,3,1,2)
a(1,2,3,2)=a(2,3,1,2)
a(2,1,2,3)=a(2,3,1,2)
a(2,1,3,2)=a(2,3,1,2)
a(3,1,1,2)=a(1,3,1,2)
a(1,3,2,1)=a(1,3,1,2)
a(3,1,2,1)=a(1,3,1,2)
a(1,2,1,3)=a(1,3,1,2)
a(2,1,1,3)=a(1,3,1,2)
a(1,2,3,1)=a(1,3,1,2)
a(2,1,3,1)=a(1,3,1,2)
return
end subroutine symmetry4d
!***************************************************************************************************


!***************************************************************************************************
  subroutine krondel(del,n)
!***************************************************************************************************
implicit none
integer 		:: i,j,n
real 			:: del(n,n)

do i = 1,n
	do j = 1,n
    	if (i.eq.j) then
        	del(i,j) = 1.0
        elseif (i.ne.j) then
            del(i,j) = 0.0
        endif
    enddo
enddo
return
end subroutine krondel
!***************************************************************************************************

! These routines are from Wookey's Ematrix_Utils library
!***************************************************************************************************
   subroutine fcross(V1,V2,fcr)
!***************************************************************************************************
!  Vector product of 2 3-vectors
real :: V1(3),V2(3)
real :: fcr(3)
            
fcr(1) = V1(2)*V2(3) - V1(3)*V2(2)
fcr(2) = V1(3)*V2(1) - V1(1)*V2(3)
fcr(3) = V1(1)*V2(2) - V1(2)*V2(1)

return
end subroutine fcross
!***************************************************************************************************
   
   
!***************************************************************************************************
   subroutine VrotateY(V,theta,VR)                                              
!***********************************************************************                                                                            
!  Rotate vector V clockwise theta degrees around the Y axis -> VR              
real :: VRtmp(1,3),VR(3)                                              
real :: V(3),Vtmp(1,3),theta,th                                       
real :: R(3,3) 
real,parameter		:: pi = 3.141592654
                                                                                
!  ** convert to radians                                                        
th = theta*pi/180 ;                                                       
!  ** construct rotation matrix                                                 
R(1,1) =  cos(th) ; R(1,2) = 0.0 ;      R(1,3) = -sin(th) ;               
R(2,1) =  0.0 ;     R(2,2) = 1.0 ;      R(2,3) = 0.0 ;                    
R(3,1) =  sin(th) ; R(3,2) = 0.0 ;      R(3,3) = cos(th) ;                
!  ** apply rotation by matrix multiplication                                   
Vtmp(1,:) = V(:)                                                          
VRtmp = matmul(Vtmp,R) ;                                                  
VR(:) = VRtmp(1,:)                                                        
                                                                                
return                                                                    
end subroutine VrotateY                                                      
!***************************************************************************************************
             
             
!***************************************************************************************************
   subroutine VrotateZ(V,theta,VR)
!***************************************************************************************************
!  Rotate vector V clockwise theta degrees around the X axis -> VR
real :: VRtmp(1,3),VR(3)
real :: V(3),Vtmp(1,3),theta,th
real :: R(3,3)
real,parameter		:: pi = 3.141592654

!  ** convert to radians      
th = theta*pi/180;
!  ** construct rotation matrix            
R(1,1) =  cos(th) ; R(1,2) = sin(th) ;  R(1,3) = 0.0 ; 
R(2,1) = -sin(th) ; R(2,2) = cos(th) ;  R(2,3) = 0.0 ; 
R(3,1) = 0.0 ;      R(3,2) = 0.0 ;      R(3,3) = 1.0 ; 
!  ** apply rotation by matrix multiplication 
Vtmp(1,:) = V(:) 
VRtmp = matmul(Vtmp,R) ;
VR(:) = VRtmp(1,:) 
 
return
end subroutine VrotateZ
!***************************************************************************************************


! Versions of Wookey's code modified by JPV
!***************************************************************************************************
   subroutine anisvelocity(c,rho,paz,pdip,vp,vsf,vss,pol)
!***************************************************************************************************
integer				:: i,j
real,dimension(6,6)	:: c
real,dimension(3)	:: v,xi
real,dimension(3,3)	:: eigvec
real				:: rho,paz,pdip,vp,vsf,vss,pol
real				:: p(3), s1(3), s2(3), s1n(3), s1p(3), s1pr(3), s1prr(3)
real,parameter		:: pi = 3.141592654

call cart2(+1,pdip,paz,xi)   ! Generate the propagation normals

call velo(xi,rho,c,v,eigvec)   ! Compute velocity


vp=v(1)
vsf=v(2)
vss=v(3)

! Compute SWS polarisation in ray frame (based on Wookey's ematrixUtils library) 
p(:)=eigvec(1,:)
s1(:)=eigvec(2,:)
s2(:)=eigvec(3,:)
call fcross(xi,s1,s1n)
call fcross(xi,s1n,s1p)
call VrotateZ(s1p,paz,s1pr)
call VrotateY(s1pr,pdip,s1prr)
pol = (atan2(s1prr(2),s1prr(3)) * 180/pi)

return
end subroutine anisvelocity
!***************************************************************************************************


!***************************************************************************************************
   subroutine velo(x,rho,c,v,eigvec)
!***************************************************************************************************
integer					:: i,j,k,l,m,n,iop,nop,inop,ijkl(3,3)
real,dimension(6,6)		:: c
real,dimension(3,3)		:: eigvec,ei,t
real,dimension(3)		:: v,x,eval
real					:: valx,valuex

data ((ijkl(i,j),j=1,3),i=1,3)/1,6,5,6,2,4,5,4,3/

do i=1,3
	do k=1,3
		t(i,k)=0.0
		do j=1,3
			do l=1,3
				m=ijkl(i,j)
				n=ijkl(k,l)
				t(i,k)=t(i,k)+c(m,n)*x(j)*x(l)
			enddo
		enddo
	enddo
enddo
call jacobi(t,3,3,eval,ei,nrot)
do i=1,3
	v(i)=sqrt(eval(i)/rho)
enddo

call eigsrt(v,ei,3,3)

do i=1,3
	do j=1,3
      	eigvec(i,j)=ei(j,i)
	enddo
enddo

return
end subroutine velo
!***************************************************************************************************


!***************************************************************************************************
   subroutine cart2(irev,inc,az,x)
!***************************************************************************************************
integer					:: i,irev
real,dimension(3)		:: x
real					:: rad,inc,az,caz,saz,cinc,sinc,r
data rad/0.0174532/

caz=cos(az*rad)
saz=sin(az*rad)
cinc=cos(inc*rad)
sinc=sin(inc*rad)
x(1)=caz*cinc
x(2)=-saz*cinc
x(3)=sinc
r=sqrt(x(1)*x(1)+x(2)*x(2)+x(3)*x(3))
do i=1,3
        x(i)=x(i)/r
enddo
if(irev.eq.-1) then
	do i=1,3
    	x(i)=-x(i)
    enddo
endif
return
end subroutine cart2
!***************************************************************************************************

   
! From here down are numerical recipes
! Mess with at your own risk
!***************************************************************************************************
   subroutine ludcmp(a,n,np,indx,d)
!***************************************************************************************************
integer					:: n,np,indx(n),itemp
integer					:: i,imax,j,k
real					:: d,a(np,np)
real,parameter			:: tiny=1.0e-20
real					:: aamax,dum,sum,vv(500)

      d=1.d0
      do 12 i=1,n
         aamax=0.
         do 11 j=1,n
            if (abs(a(i,j)).gt.aamax) aamax=abs(a(i,j))
 11      continue
         if (aamax.eq.0.) then 
         	write(*,*)'SINGULAR MATRIX IN LUDCMP'
         	read(*,*)itemp
         	if (itemp.eq.1)stop
         endif
         vv(i)=1./aamax
 12   continue
      do 19 j=1,n
         do 14 i=1,j-1
            sum=a(i,j)
            do 13 k=1,i-1
               sum=sum-a(i,k)*a(k,j)
 13         continue
            a(i,j)=sum
 14      continue
         aamax=0.
         do 16 i=j,n
            sum=a(i,j)
            do 15 k=1,j-1
               sum=sum-a(i,k)*a(k,j)
 15         continue
            a(i,j)=sum
            dum=vv(i)*abs(sum)
            if (dum.ge.aamax) then
               imax=i
               aamax=dum
            endif
 16      continue
         if (j.ne.imax)then
            do 17 k=1,n
               dum=a(imax,k)
               a(imax,k)=a(j,k)
               a(j,k)=dum
 17         continue
            d=-d
            vv(imax)=vv(j)
         endif
         indx(j)=imax
         if(a(j,j).eq.0.)a(j,j)=tiny
         if(j.ne.n)then
            dum=1.d0/a(j,j)
            do 18 i=j+1,n
               a(i,j)=a(i,j)*dum
 18         continue
         endif
 19   continue

return
end subroutine ludcmp
!***************************************************************************************************


!***************************************************************************************************
   subroutine lubksb(a,n,np,indx,b)
!***************************************************************************************************

integer				:: n,np,indx(n)
integer				:: i,ii,j,ll
real				:: a(np,np),b(n)
real				:: sum

      ii=0
      do 12 i=1,n
         ll=indx(i)
         sum=b(ll)
         b(ll)=b(i)
         if (ii.ne.0)then
            do 11 j=ii,i-1
               sum=sum-a(i,j)*b(j)
 11         continue
         else if (sum.ne.0.) then
            ii=i
         endif
         b(i)=sum
 12   continue
      do 14 i=n,1,-1
         sum=b(i)
         do 13 j=i+1,n
            sum=sum-a(i,j)*b(j)
 13      continue
         b(i)=sum/a(i,i)
 14   continue

return
end subroutine lubksb
!***************************************************************************************************


!***************************************************************************************************
   SUBROUTINE JACOBI(A,N,NP,D,V,NROT)
!  Numerical recipes
!***************************************************************************************************

      IMPLICIT REAL(A-H,O-Z)
      PARAMETER (NMAX=100)
      DIMENSION A(NP,NP),D(NP),V(NP,NP),B(NMAX),Z(NMAX)
      DO 12 IP=1,N
        DO 11 IQ=1,N
          V(IP,IQ)=0.
11      CONTINUE
        V(IP,IP)=1.
12    CONTINUE
      DO 13 IP=1,N
        B(IP)=A(IP,IP)
        D(IP)=B(IP)
        Z(IP)=0.
13    CONTINUE
      NROT=0
      DO 24 I=1,50
        SM=0.
        DO 15 IP=1,N-1
          DO 14 IQ=IP+1,N
            SM=SM+ABS(A(IP,IQ))
14        CONTINUE
15      CONTINUE
        IF(SM.EQ.0.)RETURN
        IF(I.LT.4)THEN
          TRESH=0.2*SM/N**2
        ELSE
          TRESH=0.
        ENDIF
        DO 22 IP=1,N-1
          DO 21 IQ=IP+1,N
            G=100.*ABS(A(IP,IQ))
            IF((I.GT.4).AND.(ABS(D(IP))+G.EQ.ABS(D(IP))).AND.(ABS(D(IQ))+G.EQ.ABS(D(IQ))))THEN
              A(IP,IQ)=0.
            ELSE IF(ABS(A(IP,IQ)).GT.TRESH)THEN
              H=D(IQ)-D(IP)
              IF(ABS(H)+G.EQ.ABS(H))THEN
                T=A(IP,IQ)/H
              ELSE
                THETA=0.5*H/A(IP,IQ)
                T=1./(ABS(THETA)+SQRT(1.+THETA**2))
                IF(THETA.LT.0.)T=-T
              ENDIF
              C=1./SQRT(1+T**2)
              S=T*C
              TAU=S/(1.+C)
              H=T*A(IP,IQ)
              Z(IP)=Z(IP)-H
              Z(IQ)=Z(IQ)+H
              D(IP)=D(IP)-H
              D(IQ)=D(IQ)+H
              A(IP,IQ)=0.
              DO 16 J=1,IP-1
                G=A(J,IP)
                H=A(J,IQ)
                A(J,IP)=G-S*(H+G*TAU)
                A(J,IQ)=H+S*(G-H*TAU)
16            CONTINUE
              DO 17 J=IP+1,IQ-1
                G=A(IP,J)
                H=A(J,IQ)
                A(IP,J)=G-S*(H+G*TAU)
                A(J,IQ)=H+S*(G-H*TAU)
17            CONTINUE
              DO 18 J=IQ+1,N
                G=A(IP,J)
                H=A(IQ,J)
                A(IP,J)=G-S*(H+G*TAU)
                A(IQ,J)=H+S*(G-H*TAU)
18            CONTINUE
              DO 19 J=1,N
                G=V(J,IP)
                H=V(J,IQ)
                V(J,IP)=G-S*(H+G*TAU)
                V(J,IQ)=H+S*(G-H*TAU)
19            CONTINUE
              NROT=NROT+1
            ENDIF
21        CONTINUE
22      CONTINUE
        DO 23 IP=1,N
          B(IP)=B(IP)+Z(IP)
          D(IP)=B(IP)
          Z(IP)=0.
23      CONTINUE
24    CONTINUE
!      PAUSE '50 iterations should never happen'
      RETURN
      END SUBROUTINE JACOBI
!***************************************************************************************************
      

!***************************************************************************************************
      SUBROUTINE TRED(A,N,NP,D,E)
!     Numerical Recipes subroutine.
!***************************************************************************************************

      REAL A(NP,NP),D(NP),E(NP)

      IF(N.GT.1)THEN
         DO 18 I=N,2,-1  
            L=I-1
            H=0.
            SCALE=0.
            IF(L.GT.1)THEN
               DO 11 K=1,L
                  SCALE=SCALE+ABS(A(I,K))
 11            CONTINUE
               IF(SCALE.EQ.0.)THEN
                  E(I)=A(I,L)
               ELSE
                  DO 12 K=1,L
                     A(I,K)=A(I,K)/SCALE
                     H=H+A(I,K)**2
 12               CONTINUE
                  F=A(I,L)
                  G=-SIGN(SQRT(H),F)
                  E(I)=SCALE*G
                  H=H-F*G
                  A(I,L)=F-G
                  F=0.
                  DO 15 J=1,L
                     A(J,I)=A(I,J)/H
                     G=0.
                     DO 13 K=1,J
                        G=G+A(J,K)*A(I,K)
 13                  CONTINUE
                     IF(L.GT.J)THEN
                        DO 14 K=J+1,L
                           G=G+A(K,J)*A(I,K)
 14                     CONTINUE
                     ENDIF
                     E(J)=G/H
                     F=F+E(J)*A(I,J)
 15               CONTINUE
                  HH=F/(H+H)
                  DO 17 J=1,L
                     F=A(I,J)
                     G=E(J)-HH*F
                     E(J)=G
                     DO 16 K=1,J
                        A(J,K)=A(J,K)-F*E(K)-G*A(I,K)
 16                  CONTINUE
 17               CONTINUE
               ENDIF
            ELSE
               E(I)=A(I,L)
            ENDIF
            D(I)=H
 18      CONTINUE
      ENDIF
      D(1)=0.
      E(1)=0.
      DO 23 I=1,N
         L=I-1
         IF(D(I).NE.0.)THEN
            DO 21 J=1,L
               G=0.
               DO 19 K=1,L
                  G=G+A(I,K)*A(K,J)
 19            CONTINUE
               DO 20 K=1,L
                  A(K,J)=A(K,J)-G*A(K,I)
 20            CONTINUE
 21         CONTINUE
         ENDIF
         D(I)=A(I,I)
         A(I,I)=1.
         IF(L.GE.1)THEN
            DO 22 J=1,L
               A(I,J)=0.
               A(J,I)=0.
 22         CONTINUE
         ENDIF
 23   CONTINUE
      RETURN
      END SUBROUTINE TRED
!***************************************************************************************************


!***************************************************************************************************
      SUBROUTINE TQLI(D,E,N,NP,Z)
!     Numerical Recipes subroutine.
!***************************************************************************************************

      REAL D(NP),E(NP),Z(NP,NP)

      IF (N.GT.1) THEN
         DO 11 I=2,N
            E(I-1)=E(I)
 11      CONTINUE
         E(N)=0.
         DO 15 L=1,N
            ITER=0
 1          DO 12 M=L,N-1
               DD=ABS(D(M))+ABS(D(M+1))
               IF (ABS(E(M))+DD.EQ.DD) GO TO 2
 12         CONTINUE
            M=N
 2          IF(M.NE.L)THEN
               IF(ITER.EQ.30)then
               	write(*,*)'too many iterations'
               	stop
               endif
               ITER=ITER+1
               G=(D(L+1)-D(L))/(2.*E(L))
               R=SQRT(G**2+1.)
               G=D(M)-D(L)+E(L)/(G+SIGN(R,G))
               S=1.
               C=1.
               P=0.
               DO 14 I=M-1,L,-1
                  F=S*E(I)
                  B=C*E(I)
                  IF(ABS(F).GE.ABS(G))THEN
                     C=G/F
                     R=SQRT(C**2+1.)
                     E(I+1)=F*R
                     S=1./R
                     C=C*S
                  ELSE
                     S=F/G
                     R=SQRT(S**2+1.)
                     E(I+1)=G*R
                     C=1./R  
                     S=S*C
                  ENDIF
                  G=D(I+1)-P
                  R=(D(I)-G)*S+2.*C*B
                  P=S*R
                  D(I+1)=G+P
                  G=C*R-B
                  DO 13 K=1,N
                     F=Z(K,I+1)
                     Z(K,I+1)=S*Z(K,I)+C*F
                     Z(K,I)=C*Z(K,I)-S*F
 13               CONTINUE
 14            CONTINUE
               D(L)=D(L)-P
               E(L)=G
               E(M)=0.
               GO TO 1
            ENDIF
 15      CONTINUE
      ENDIF
      RETURN
      END SUBROUTINE TQLI
!***************************************************************************************************


!***************************************************************************************************
      SUBROUTINE EIGSRT(D,V,N,NP)
!     Numerical Recipes subroutine.
!***************************************************************************************************

      REAL D(NP),V(NP,NP)
      DO 13 I=1,N-1
        K=I
        P=D(I)
        DO 11 J=I+1,N
          IF(D(J).GE.P)THEN
            K=J
            P=D(J)
          ENDIF
11      CONTINUE
        IF(K.NE.I)THEN
          D(K)=D(I)
          D(I)=P
          DO 12 J=1,N
            P=V(J,I)
            V(J,I)=V(J,K)
            V(J,K)=P
12        CONTINUE
        ENDIF
13    CONTINUE

      RETURN
      END SUBROUTINE EIGSRT
!***************************************************************************************************
