!===============================================================================
!-------------------------------------------------------------------------------
!
!  Fortran 90/95 Source Code File
!
!-------------------------------------------------------------------------------
!===============================================================================
!
!  PROGRAM : RandomNumbers
!  VERSION : 4.21
!  CVS: $Revision: 1.1.1.1 $ $Date: 2008/11/27 10:48:20 $
!
!  (C) James Wookey
!  Department of Earth Sciences, University of Bristol
!  Wills Memorial Building, Queen's Road, Bristol, BR8 1RJ, UK
!  j.wookey@bristol.ac.uk
!
!-------------------------------------------------------------------------------
!
!   Module providing random number generation capacity. Note, double precision
!   random numbers generated by this code are simply 8 byte conversions of a 
!   4-byte random, not true random 8-byte widths. 
!
!===============================================================================

   module RandomNumbers 

!===============================================================================
   implicit none

!  ** publically accessible routines
      public :: RandomR8,RandomR4,RN_dateseed

!  ** deny access to core routines
      private :: RN_ran1,RN_initialise

!  ** define a long (32 bit) integer and 32 bit real      
!      integer, parameter, private :: i4 = selected_int_kind(9) ;
!      integer, parameter, private :: r4 = selected_real_kind(6,37) ;
!      integer, parameter, private :: r8 = selected_real_kind(15,307) ; 

!  ** flag to check whether the random number generator has been seeded. 
      integer, parameter, private :: RN_initflag_setval = 11075
      integer, private :: RN_initflag
      integer, private :: RN_seed
      
!===============================================================================
!
!  ** MODULE SUBROUTINES
!
!===============================================================================

   CONTAINS

!===============================================================================
   function RandomR4()
!===============================================================================
!
!     Return a single random number. 
!
      implicit none
      real*4:: RandomR4
      
      if (RN_initflag/=RN_initflag_setval) call RN_initialise()
         
      RandomR4 = RN_ran1(RN_seed)
         
      
      return
   end function RandomR4
!===============================================================================

!===============================================================================
   function RandomInt(imin,imax)
!===============================================================================
!
!     Return a single random integer between imin and imax 
!
      implicit none
      integer :: RandomInt
      integer :: imin,imax
      real*4:: RandomR4,tmp
      
      if (RN_initflag/=RN_initflag_setval)call RN_initialise()
      
      RandomR4 = RN_ran1(RN_seed)
      tmp = -0.5+real(imin)+(RandomR4*real(imax-imin+1))
      RandomInt = nint(tmp)   

      return
   end function RandomInt
!===============================================================================


!===============================================================================
   function RandomR8()
!===============================================================================
!
!     Return a single random number. 
!
      implicit none
      real*8:: RandomR8
      
      if (RN_initflag/=RN_initflag_setval) call RN_initialise()
         
      RandomR8 = dble(RN_ran1(RN_seed))  
         
      
      return
   end function RandomR8
!===============================================================================

!===============================================================================
   subroutine RN_initialise()
!===============================================================================
!
!  Initiate random number generator
!  
!  NOTE: Requires Numerical Recipes function ran1, or appropriate
!  substitute. Numerical Recipes functions are NOT freeware and 
!  are therefore NOT distributed with this code. If no appropriate
!  substitute is available, this subroutine can be commented out.
!
      implicit none
      integer i
      real dum

!  ** seed and burn in the random number generator
      RN_seed = RN_dateseed()
      
      dum = RN_ran1(-RN_seed) ! initialise ran1
      do i=1,10
          dum = RN_ran1(RN_seed)
      enddo      

!  ** set the flag
      RN_initflag=RN_initflag_setval
               
      return
   end subroutine RN_initialise
!===============================================================================

!===============================================================================
   function RN_dateseed()
!===============================================================================
!
!  Generate a random number seed using the date (day number) and time
!  Updated to use the date_and_time
!
      implicit none
      integer :: dd,mm,yy ! dd,mm,yy
      integer :: dayno,nsec
      integer :: hh,mi,se,ms      

      integer :: RN_dateseed ! random number seed 
      character :: dtsdstr*8
      character :: datestr*8,timestr*10

!  ** get the time and date      
      call date_and_time(datestr,timestr)
      read(datestr,'(i4.4,i2.2,i2.2)') yy,mm,dd
      read(timestr,'(i2.2,i2.2,i2.2,1x,i3.3)') hh,mi,se,ms

!  ** get a day (in year) number
!      call f90sac_ymd2jd(yy,mm,dd,dayno)   
      
      RN_dateseed = (hh*3600 +mi*60 +se) * 1000 + ms
      
      return
   end function RN_dateseed
!===============================================================================

!===============================================================================
   function RN_ran1(idum)
!===============================================================================
      implicit none
      integer idum,ia,im,iq,ir,ntab,ndiv
      real RN_ran1,am,eps,rnmx
      parameter (ia=16807,im=2147483647,am=1./im,iq=127773,ir=2836, &
      ntab=32,ndiv=1+(im-1)/ntab,eps=1.2e-7,rnmx=1.-eps)
      integer j,k,iv(ntab),iy
      save iv,iy
      data iv /ntab*0/, iy /0/
      if (idum.le.0.or.iy.eq.0) then
        idum=max(-idum,1)
        do 11 j=ntab+8,1,-1
          k=idum/iq
          idum=ia*(idum-k*iq)-ir*k
          if (idum.lt.0) idum=idum+im
          if (j.le.ntab) iv(j)=idum
11      continue
        iy=iv(1)
      endif
      k=idum/iq
      idum=ia*(idum-k*iq)-ir*k
      if (idum.lt.0) idum=idum+im
      j=1+iy/ndiv
      iy=iv(j)
      iv(j)=idum
      RN_ran1=min(am*iy,rnmx)
      return
   end function RN_ran1
!  (c) copr. 1986-92 numerical recipes software *%&&,1{.
!===============================================================================

!===============================================================================
   end module RandomNumbers
!===============================================================================
!  END OF RandomNumbers module
!===============================================================================
   