function [C] = effev(varargin)
% function [C] = effev(varargin)
%
% Compute the effective stiffness of a rock containing compliant fractures.
% A number of options are available, including Hudson LF, HF or
% freq-dependent fractures, or fractures defined by Zn/Zt. Requires an
% input stiffness of the background rock, which can be defined as isotropic
% by velocities or stiffness moduli, as VTI with Thomsen parameters, or by
% a pre-specified stiffness tensor with 21 independent components
%
% Dependencies: requires that the MSAT toolbox is installed
%
% INPUT Options:
%   'fracture' = structure containing the fracture set parameters. These
%                   must include:
%                       frac.type = type of fracture ('hudsonlf','hudsonhf',
%                       'hudsonfd' or 'znzt'
%                       frac.den = fracture density
%                       frac.strike = fracture strike
%                       frac.dip = fracture dip
%                   where frac.type = znzt
%                       frac.znzt = zn/zt ratio must be specified
%                   where frac.type = hudsonhf
%                       frac.asp = fracture aspect ratio must be specified 
%                       'kfluid' = fluid bulk mod must be specified
%                   where frac.type = hudsonfd
%                       frac.rad = fracture radius
%                       frac.asp = fracture aspect ratio
%                       'visc' = fluid viscosity
%                       'kfluid' = fluid bulk mod
%                       'permeability' = permeability
%                       'porosity' = porosity
%                       'frequency' = seismic wave frequency must be specified
%
% DEFINING the BACKGROUND STIFFNESS HAS A NUMBER OF OPTIONS:
%   Define a background stiffness tensor:
%       'cin' = 6x6 tensor giving background stiffness
%   Define background velocities:
%       'vp' = P wave velocity of background                      
%       'vs' = S wave velocity of background
%   Define background moduli:
%       'Pmod' = P wave modulus
%       'mu' = shear wave modulus
%   Add in Thomsen's parameters:
%       'eps' = epsilon
%       'gamma' = gamma
%       'delta' = delta
%   'rho' = density can also be specified (defaults to 2000)
% 
% FLUID SUBSITIUTION CAN ALSO BE INCLUDED:
%   'gassmann' = tag to substitute fluid
% IF FS IS PERFORMED, THE FOLLOWING MUST BE DEFINED:
%   'kfluid' = fluid bulk modulus, or
%   'vfluid' = fluid velocity
%   'rfluid' = fluid density 
%   'mmod' = mineral bulk modulus
%   'porosity' = porosity
%
% Written by J.P. Verdon, University of Bristol, 2014
%

% process the optional arguments
iarg = 1 ;
nfrac = 0;
ifluid = 0;
while iarg <= (length(varargin))
    switch lower(varargin{iarg})
        case 'fracture'
            nfrac = nfrac+1;
            fracture{nfrac} = varargin{iarg+1};
            iarg = iarg + 2;
        case 'porosity'
            por = varargin{iarg+1};
            iarg = iarg + 2;
        case 'permeability'
            perm = varargin{iarg+1};
            iarg = iarg + 2;
        case 'frequency'
            freq = varargin{iarg+1};
            iarg = iarg + 2; 
        case 'cin'
            C = varargin{iarg+1};
            iarg = iarg + 2;
        case 'vp'
            vp = varargin{iarg+1};
            iarg = iarg + 2; 
        case 'vs'
            vs = varargin{iarg+1};
            iarg = iarg + 2; 
        case 'rho'
            rho = varargin{iarg+1};
            iarg = iarg + 2;
        case 'eps'
            eps = varargin{iarg+1};
            iarg = iarg + 2;
        case 'gamma'
            gam = varargin{iarg+1};
            iarg = iarg + 2;
        case 'delta'
            del = varargin{iarg+1};
            iarg = iarg + 2;
        case 'mu'
            mu = varargin{iarg+1};
            iarg = iarg + 2;
        case 'Pmod'
            pmod = varargin{iarg+1};
            iarg = iarg + 2;    
        case 'gassmann'
            ifluid = 1;
            iarg = iarg+1;
        case 'kfluid'
            kfl = varargin{iarg+1};
            iarg = iarg + 2; 
        case 'vfluid'
            vfl = varargin{iarg+1};
            iarg = iarg + 2;
        case 'rfluid'
            rfl = varargin{iarg+1};
            iarg = iarg + 2;   
        case 'mmod'
            mmod = varargin{iarg+1};
            iarg = iarg + 2;  
        case 'visc'
            visc = varargin{iarg+1};
            iarg = iarg + 2;
      otherwise 
         error('EFFEV:UnknownOption',...
            ['Unknown option: ' varargin{iarg}]) ;   
      
    end
end


% Compute P and S wave velocities if not set
if ~exist('rho','var')
    rho = 2000;
end
if ~exist('C','var')
    if ~exist('vp','var')
        if exist('pmod','var')
            vp = sqrt(pmod/rho);
        else
            error('EFFEV:Inputs','P wave velocity or modulus not defined')
        end
    end
    if ~exist('vs','var')
        if exist('mu','var')
            vs = sqrt(mu/rho);
        else
            error('EFFEV:Inputs','P wave velocity or modulus not defined')
        end
    end
 
    C = MS_iso(vp/1000,vs/1000,rho);
    
    if exist('eps','var') || exist('gam','var') || exist('del','var')
        [C] = MS_TI(vp/1000,vs/1000,rho,eps,gam,del,'thomsen');
    end
    
    C = C * 1e9;
    
end



S = inv(C);
nu = MS_poisson( C, [1 0 0], [0 1 0]);
h = (3.0*(1.0/S(1,1))*(2.0 - nu)) / (32.0*(1-nu^2.0));
    

% Add dry fractures
for i = 1:nfrac
    if strcmpi(fracture{i}.type,'hudsonlf')
        C = hudsonlf(C,fracture{i});
    end
end



% Saturate rock with fluids
if ifluid == 1;
    if ~exist('kfl','var')
        kfl = vfl*vfl*rfl;
    end
        
    C = galvin(C,por,mmod,kfl,rfl,rho);
end


% Add wet fractures
for i = 1:nfrac
    if strcmpi(fracture{i}.type,'hudsonhf')
        C = hudsonhf(C,fracture{i},kfl);
    elseif strcmpi(fracture{i}.type,'hudsonfd')
        C = hudsonfd(C,fracture{i},kfl,visc,perm,por,freq);
    elseif strcmpi(fracture{i}.type,'znzt')
        C = znzt_frac(C,fracture{i},h);
    end
end

end  % End EFFEV






function [C,rho] = galvin(C,por,mmod,kfl,rfl,rho)
for i=1:6
	if i <= 3
        e(i) = 1;
	end
    if i > 3
        e(i) = 0;
    end
end

al = 1 / (por * ((1 / kfl) - (1 / mmod)));
	  
for i=1:6
	b(i) = e(i) - ((C(1,i) + C(2,i) + C(3,i)) / (3 * mmod));
end  
	  
dstar = 1 + al * (b(1) + b(2) + b(3)) / (3 * mmod);
	  
for i=1:6
	for j=1:6
		C(i,j) = C(i,j) + (al * b(i) * b(j) / dstar);
	end
end

% Sort out densities
rho=rho+rfl*por;
end


function C = hudsonlf(C,fracture)
zn=(4/3)*(fracture.den/C(6,6))*(C(1,1)/(C(1,1)-C(6,6)));
zt=(16/3)*(fracture.den/C(4,4))*(C(3,3)/(3*C(3,3)-2*C(4,4)));

C = SandS(C,zn,zt,fracture.strike,fracture.dip);
end


function C = hudsonhf(C,fracture,kfl)	  
ak=(kfl/(pi*fracture.asp*C(6,6)))*(C(1,1)/(3*C(3,3)-2*C(4,4)));

zn=(4/3)*(fracture.den/C(6,6))*(C(1,1)/(C(1,1)-C(6,6)))*(1/(1+ak));
zt=(16/3)*(fracture.den/C(4,4))*(C(3,3)/(3*C(3,3)-2*C(4,4)));

C = SandS(C,zn,zt,fracture.strike,fracture.dip);
end


function C = hudsonfd(C,fracture,kfl,visc,perm,por,freq)
frap=fracture.rad*fracture.asp;
aj=sqrt(0.5*por*kfl*perm/(visc*freq));
ak=(kfl/(pi*fracture.asp*C(6,6)))*(C(1,1)/(3*C(3,3)-2*C(4,4)));
ak=ak*1/(1+(3*aj/(2*frap)));

zn=(4/3)*(fracture.den/C(6,6))*(C(1,1)/(C(1,1)-C(6,6)))*(1/(1+ak));
zt=(16/3)*(fracture.den/C(4,4))*(C(3,3)/(3*C(3,3)-2*C(4,4)));
	
C = SandS(C,zn,zt,fracture.strike,fracture.dip);
	
end


function C = znzt_frac(C,fracture,h)
zt = fracture.den/h;
zn = zt*fracture.znzt;

C = SandS(C,zn,zt,fracture.strike,fracture.dip);

end


function C = SandS(C,zn,zt,strike,dip)
alp = 90 - strike;
bet = dip;
x(1) = cosd(bet)*cosd(alp);
x(2) = cosd(bet)*sind(alp);
x(3) = sind(bet);

kdel = [1 0 0 ; 0 1 0 ; 0 0 1];

for i = 1:3
    for j = 1:3
    	z(i,j) = zt*kdel(i,j) + (zn-zt)*x(i)*x(j);
    end
end

ssft = zeros(3,3,3,3);
ssf = zeros(3,3,3,3);

for i=1:3
	for j=1:3
        for k=1:3
            for l=1:3
            	ssf(i,j,k,l)= z(i,k)*x(l)*x(j)+z(j,k)*x(l)*x(i)+z(i,l)*x(k)*x(j)+z(j,l)*x(k)*x(i);
                ssft(i,j,k,l) = ssft(i,j,k,l) + 0.25*ssf(i,j,k,l);
            end
        end
	end
end


sf = MS_sijkl2sij(ssft);
sb = inv(C);
s = sb + sf;
C = inv(s);

end






